"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.applyCouponSchema = exports.validateCouponSchema = exports.updateCouponSchema = exports.couponSchema = exports.updateAmenitySchema = exports.amenitySchema = exports.updateExternalCalendarSchema = exports.externalCalendarSchema = exports.replyContactMessageSchema = exports.updateContactMessageSchema = exports.contactMessageSchema = exports.updatePaymentStatusSchema = exports.manualPaymentSchema = exports.paymentSchema = exports.bookingRulesSchema = exports.updateBookingSchema = exports.bookingSchema = exports.updateGuestSchema = exports.guestSchema = exports.updateRoomTypeSchema = exports.roomTypeSchema = exports.updateRoomSchema = exports.roomSchema = exports.updateUserSchema = exports.userSchema = exports.updatePasswordSchema = exports.registerSchema = exports.loginSchema = void 0;
const zod_1 = require("zod");
// Validation schema for authentication
exports.loginSchema = zod_1.z.object({
    body: zod_1.z.object({
        email: zod_1.z.string().email('Invalid email format'),
        password: zod_1.z.string().min(6, 'Password must be at least 6 characters'),
    }),
});
exports.registerSchema = zod_1.z.object({
    body: zod_1.z.object({
        firstName: zod_1.z.string().min(1, 'First name is required'),
        lastName: zod_1.z.string().min(1, 'Last name is required'),
        email: zod_1.z.string().email('Invalid email format'),
        password: zod_1.z.string().min(6, 'Password must be at least 6 characters'),
        role: zod_1.z.enum(['admin', 'manager', 'staff']).optional(),
    }),
});
exports.updatePasswordSchema = zod_1.z.object({
    body: zod_1.z.object({
        currentPassword: zod_1.z.string().min(1, 'Current password is required'),
        newPassword: zod_1.z.string().min(6, 'New password must be at least 6 characters'),
        passwordConfirm: zod_1.z.string().min(6, 'Password confirmation must be at least 6 characters'),
    }),
});
// Validation schema for users
exports.userSchema = zod_1.z.object({
    body: zod_1.z.object({
        firstName: zod_1.z.string().min(1, 'First name is required'),
        lastName: zod_1.z.string().min(1, 'Last name is required'),
        email: zod_1.z.string().email('Invalid email format'),
        password: zod_1.z.string().min(6, 'Password must be at least 6 characters'),
        role: zod_1.z.enum(['admin', 'manager', 'staff']).optional(),
    }),
});
exports.updateUserSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        firstName: zod_1.z.string().min(1, 'First name is required').optional(),
        lastName: zod_1.z.string().min(1, 'Last name is required').optional(),
        email: zod_1.z.string().email('Invalid email format').optional(),
        role: zod_1.z.enum(['admin', 'manager', 'staff']).optional(),
    }),
});
// Validation schema for rooms
exports.roomSchema = zod_1.z.object({
    body: zod_1.z.object({
        roomTypeId: zod_1.z.coerce.number().int().positive('Room type ID must be a positive integer'),
        number: zod_1.z.string().min(1, 'Room number is required'),
        floor: zod_1.z.string().min(1, 'Floor is required'),
        capacity: zod_1.z.coerce.number().int().positive('Capacity must be a positive integer'),
        pricePerNight: zod_1.z.coerce.number().positive('Price per night must be a positive number').optional(),
        discount: zod_1.z.coerce.number().min(0, 'Discount must be at least 0').max(100, 'Discount must be at most 100').optional(),
        status: zod_1.z.enum(['available', 'occupied', 'maintenance', 'reserved']).optional(),
        description: zod_1.z.string().optional(),
        // Images will be handled by multer middleware, not by zod
    }),
});
exports.updateRoomSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        roomTypeId: zod_1.z.coerce.number().int().positive('Room type ID must be a positive integer').optional(),
        number: zod_1.z.string().min(1, 'Room number is required').optional(),
        floor: zod_1.z.string().min(1, 'Floor is required').optional(),
        capacity: zod_1.z.coerce.number().int().positive('Capacity must be a positive integer').optional(),
        pricePerNight: zod_1.z.coerce.number().positive('Price per night must be a positive number').optional(),
        discount: zod_1.z.coerce.number().min(0, 'Discount must be at least 0').max(100, 'Discount must be at most 100').optional(),
        status: zod_1.z.enum(['available', 'occupied', 'maintenance', 'reserved']).optional(),
        description: zod_1.z.string().optional(),
        imagesToDelete: zod_1.z.array(zod_1.z.string()).optional(),
        // Images will be handled by multer middleware, not by zod
    }),
});
// Validation schema for room types
exports.roomTypeSchema = zod_1.z.object({
    body: zod_1.z.object({
        name: zod_1.z.string().min(1, 'Name is required'),
        description: zod_1.z.string().optional(),
        basePrice: zod_1.z.number().positive('Base price must be a positive number'),
    }),
});
exports.updateRoomTypeSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        name: zod_1.z.string().min(1, 'Name is required').optional(),
        description: zod_1.z.string().optional(),
        basePrice: zod_1.z.number().positive('Base price must be a positive number').optional(),
    }),
});
// Validation schema for guests
exports.guestSchema = zod_1.z.object({
    body: zod_1.z.object({
        firstName: zod_1.z.string().min(1, 'First name is required'),
        lastName: zod_1.z.string().min(1, 'Last name is required'),
        email: zod_1.z.string().email('Invalid email format'),
        phone: zod_1.z.string().min(1, 'Phone is required'),
        address: zod_1.z.string().optional(),
        city: zod_1.z.string().optional(),
        state: zod_1.z.string().optional(),
        country: zod_1.z.string().optional(),
        postalCode: zod_1.z.string().optional(),
    }),
});
exports.updateGuestSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        firstName: zod_1.z.string().min(1, 'First name is required').optional(),
        lastName: zod_1.z.string().min(1, 'Last name is required').optional(),
        email: zod_1.z.string().email('Invalid email format').optional(),
        phone: zod_1.z.string().min(1, 'Phone is required').optional(),
        address: zod_1.z.string().optional(),
        city: zod_1.z.string().optional(),
        state: zod_1.z.string().optional(),
        country: zod_1.z.string().optional(),
        postalCode: zod_1.z.string().optional(),
    }),
});
// Validation schema for bookings
exports.bookingSchema = zod_1.z.object({
    body: zod_1.z.object({
        roomId: zod_1.z.number().int().positive('Room ID must be a positive integer'),
        guestId: zod_1.z.number().int().positive('Guest ID must be a positive integer'),
        checkInDate: zod_1.z.string().refine((date) => !isNaN(Date.parse(date)), 'Invalid check-in date format'),
        checkOutDate: zod_1.z.string().refine((date) => !isNaN(Date.parse(date)), 'Invalid check-out date format'),
        specialRequests: zod_1.z.string().optional(),
    }),
});
exports.updateBookingSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        roomId: zod_1.z.number().int().positive('Room ID must be a positive integer').optional(),
        checkInDate: zod_1.z.string().refine((date) => !isNaN(Date.parse(date)), 'Invalid check-in date format').optional(),
        checkOutDate: zod_1.z.string().refine((date) => !isNaN(Date.parse(date)), 'Invalid check-out date format').optional(),
        status: zod_1.z.enum(['confirmed', 'pending', 'cancelled', 'completed']).optional(),
        specialRequests: zod_1.z.string().optional(),
    }),
});
// Booking rules schema (for future use if needed)
exports.bookingRulesSchema = zod_1.z.object({
    body: zod_1.z.object({
        rules: zod_1.z
            .array(zod_1.z.object({
            month: zod_1.z.number().int().min(1).max(12),
            minNights: zod_1.z.number().int().min(1).max(20),
        }))
            .min(1),
    }),
});
// Validation schema for payments
exports.paymentSchema = zod_1.z.object({
    body: zod_1.z.object({
        bookingId: zod_1.z.number().int().positive('Booking ID must be a positive integer'),
        paymentMethod: zod_1.z.enum(['bank_card', 'wallet', 'e-DINAR']),
        notes: zod_1.z.string().optional(),
    }),
});
// Manual payment (admin/manager) schema
exports.manualPaymentSchema = zod_1.z.object({
    body: zod_1.z.object({
        bookingId: zod_1.z.number().int().positive('Booking ID must be a positive integer'),
        amount: zod_1.z.number().positive('Amount must be a positive number'),
        paymentMethod: zod_1.z.enum(['cash', 'credit_card', 'debit_card', 'bank_transfer', 'online']),
        notes: zod_1.z.string().optional(),
        status: zod_1.z.enum(['pending', 'completed', 'failed', 'refunded']).optional(),
    }),
});
exports.updatePaymentStatusSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        status: zod_1.z.enum(['pending', 'completed', 'failed', 'refunded']),
        notes: zod_1.z.string().optional(),
    }),
});
// Validation schema for contact messages
exports.contactMessageSchema = zod_1.z.object({
    body: zod_1.z.object({
        name: zod_1.z.string().min(1, 'Name is required'),
        email: zod_1.z.string().email('Invalid email format'),
        phone: zod_1.z.string().optional(),
        subject: zod_1.z.string().min(1, 'Subject is required'),
        message: zod_1.z.string().min(1, 'Message is required'),
    }),
});
exports.updateContactMessageSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        status: zod_1.z.enum(['new', 'read', 'replied', 'archived']).optional(),
        assignedTo: zod_1.z.number().int().positive('Assigned user ID must be a positive integer').nullable().optional(),
    }),
});
exports.replyContactMessageSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        replyMessage: zod_1.z.string().min(1, 'Reply message is required'),
    }),
});
// Validation schema for external calendars
exports.externalCalendarSchema = zod_1.z.object({
    body: zod_1.z.object({
        name: zod_1.z.string().min(1, 'Name is required'),
        source: zod_1.z.string().min(1, 'Source is required'),
        calendarUrl: zod_1.z.string().url('Invalid calendar URL'),
    }),
});
exports.updateExternalCalendarSchema = zod_1.z.object({
    params: zod_1.z.object({
        calendarId: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        name: zod_1.z.string().min(1, 'Name is required').optional(),
        source: zod_1.z.string().min(1, 'Source is required').optional(),
        calendarUrl: zod_1.z.string().url('Invalid calendar URL').optional(),
        isActive: zod_1.z.boolean().optional(),
    }),
});
// Validation schema for amenities
exports.amenitySchema = zod_1.z.object({
    body: zod_1.z.object({
        name: zod_1.z.string().min(2, 'Name must be at least 2 characters').max(100, 'Name must be at most 100 characters'),
        description: zod_1.z.string().min(10, 'Description must be at least 10 characters'),
        icon: zod_1.z.string().min(1, 'Icon is required'),
    }),
});
exports.updateAmenitySchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        name: zod_1.z.string().min(2, 'Name must be at least 2 characters').max(100, 'Name must be at most 100 characters').optional(),
        description: zod_1.z.string().min(10, 'Description must be at least 10 characters').optional(),
        icon: zod_1.z.string().min(1, 'Icon is required').optional(),
    }),
});
// Validation schema for coupons
exports.couponSchema = zod_1.z.object({
    body: zod_1.z.object({
        code: zod_1.z.string().min(3, 'Code must be at least 3 characters').max(20, 'Code must be at most 20 characters'),
        discountPercentage: zod_1.z.number().min(0, 'Discount percentage must be at least 0').max(100, 'Discount percentage must be at most 100'),
        maxDiscount: zod_1.z.number().positive('Maximum discount must be a positive number'),
        minBookingAmount: zod_1.z.number().min(0, 'Minimum booking amount must be at least 0'),
        startDate: zod_1.z.string().refine((date) => !isNaN(Date.parse(date)), 'Invalid start date format'),
        endDate: zod_1.z.string().refine((date) => !isNaN(Date.parse(date)), 'Invalid end date format'),
        isActive: zod_1.z.boolean().optional(),
        usageLimit: zod_1.z.number().int().min(1, 'Usage limit must be at least 1'),
    }),
});
exports.updateCouponSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().min(1),
    }),
    body: zod_1.z.object({
        code: zod_1.z.string().min(3, 'Code must be at least 3 characters').max(20, 'Code must be at most 20 characters').optional(),
        discountPercentage: zod_1.z.number().min(0, 'Discount percentage must be at least 0').max(100, 'Discount percentage must be at most 100').optional(),
        maxDiscount: zod_1.z.number().positive('Maximum discount must be a positive number').optional(),
        minBookingAmount: zod_1.z.number().min(0, 'Minimum booking amount must be at least 0').optional(),
        startDate: zod_1.z.string().refine((date) => !isNaN(Date.parse(date)), 'Invalid start date format').optional(),
        endDate: zod_1.z.string().refine((date) => !isNaN(Date.parse(date)), 'Invalid end date format').optional(),
        isActive: zod_1.z.boolean().optional(),
        usageLimit: zod_1.z.number().int().min(1, 'Usage limit must be at least 1').optional(),
    }),
});
exports.validateCouponSchema = zod_1.z.object({
    body: zod_1.z.object({
        code: zod_1.z.string().min(3, 'Code must be at least 3 characters'),
        bookingAmount: zod_1.z.number().positive('Booking amount must be a positive number'),
    }),
});
exports.applyCouponSchema = zod_1.z.object({
    body: zod_1.z.object({
        code: zod_1.z.string().min(3, 'Code must be at least 3 characters'),
        bookingAmount: zod_1.z.number().positive('Booking amount must be a positive number'),
    }),
});
