"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyTimedToken = exports.generateTimedToken = exports.generateSecureToken = exports.generateFeedbackToken = void 0;
const uuid_1 = require("uuid");
const crypto_1 = __importDefault(require("crypto"));
/**
 * Generate a unique feedback token for guest feedback forms
 * @returns A secure random token string
 */
const generateFeedbackToken = () => {
    // Generate a UUID and add some randomness
    const uuid = (0, uuid_1.v4)();
    const randomBytes = crypto_1.default.randomBytes(16).toString('hex');
    // Combine and hash for additional security
    const combined = `${uuid}-${randomBytes}`;
    const hash = crypto_1.default.createHash('sha256').update(combined).digest('hex');
    // Return first 32 characters for a manageable token length
    return hash.substring(0, 32);
};
exports.generateFeedbackToken = generateFeedbackToken;
/**
 * Generate a secure random token for general use
 * @param length Length of the token (default: 32)
 * @returns A secure random token string
 */
const generateSecureToken = (length = 32) => {
    return crypto_1.default.randomBytes(Math.ceil(length / 2)).toString('hex').substring(0, length);
};
exports.generateSecureToken = generateSecureToken;
/**
 * Generate a JWT-like token structure (without actual JWT signing)
 * @param payload Data to include in the token
 * @param expiresIn Expiration time in milliseconds
 * @returns A token string with embedded expiration
 */
const generateTimedToken = (payload, expiresIn) => {
    const expirationTime = Date.now() + expiresIn;
    const tokenData = {
        payload,
        exp: expirationTime,
        iat: Date.now()
    };
    const encoded = Buffer.from(JSON.stringify(tokenData)).toString('base64');
    const signature = crypto_1.default.createHash('sha256').update(encoded + process.env.JWT_SECRET).digest('hex');
    return `${encoded}.${signature}`;
};
exports.generateTimedToken = generateTimedToken;
/**
 * Verify a timed token
 * @param token The token to verify
 * @returns The payload if valid, null if invalid or expired
 */
const verifyTimedToken = (token) => {
    try {
        const [encoded, signature] = token.split('.');
        // Verify signature
        const expectedSignature = crypto_1.default.createHash('sha256').update(encoded + process.env.JWT_SECRET).digest('hex');
        if (signature !== expectedSignature) {
            return null;
        }
        // Decode and check expiration
        const tokenData = JSON.parse(Buffer.from(encoded, 'base64').toString());
        if (Date.now() > tokenData.exp) {
            return null; // Token expired
        }
        return tokenData.payload;
    }
    catch (error) {
        return null;
    }
};
exports.verifyTimedToken = verifyTimedToken;
