"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.syncExternalReviews = exports.updateRoomReviews = exports.fetchBookingReviews = exports.fetchAirbnbReviews = void 0;
const axios_1 = __importDefault(require("axios"));
const models_1 = __importDefault(require("../models"));
/**
 * Fetches reviews from Airbnb API
 * @param listingId The Airbnb listing ID
 * @param apiKey The Airbnb API key
 */
const fetchAirbnbReviews = (listingId, apiKey) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const response = yield axios_1.default.get(`https://api.airbnb.com/v2/reviews`, {
            params: {
                listing_id: listingId,
                role: 'host',
                _limit: 100,
            },
            headers: {
                'Authorization': `Bearer ${apiKey}`,
            },
        });
        return response.data.reviews.map((review) => ({
            id: review.id.toString(),
            rating: review.rating,
            comment: review.comments,
            guestName: `${review.reviewer.first_name} ${review.reviewer.last_name}`,
            guestEmail: review.reviewer.email,
            reviewDate: new Date(review.created_at),
            source: 'Airbnb',
        }));
    }
    catch (error) {
        console.error('Error fetching Airbnb reviews:', error);
        throw new Error(`Failed to fetch Airbnb reviews: ${error.message}`);
    }
});
exports.fetchAirbnbReviews = fetchAirbnbReviews;
/**
 * Fetches reviews from Booking.com API
 * @param hotelId The Booking.com hotel ID
 * @param apiKey The Booking.com API key
 */
const fetchBookingReviews = (hotelId, apiKey) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const response = yield axios_1.default.get(`https://distribution-xml.booking.com/json/bookings`, {
            params: {
                hotel_id: hotelId,
                review_type: 'all',
                rows: 100,
            },
            headers: {
                'Authorization': `Basic ${apiKey}`,
            },
        });
        return response.data.result.map((review) => ({
            id: review.review_id.toString(),
            rating: review.review_score,
            comment: review.review_text,
            guestName: review.guest_name,
            guestEmail: review.guest_email,
            reviewDate: new Date(review.review_date),
            source: 'Booking.com',
        }));
    }
    catch (error) {
        console.error('Error fetching Booking.com reviews:', error);
        throw new Error(`Failed to fetch Booking.com reviews: ${error.message}`);
    }
});
exports.fetchBookingReviews = fetchBookingReviews;
/**
 * Updates room reviews based on external platform reviews
 * @param roomId The ID of the room
 * @param reviews Array of external reviews
 */
const updateRoomReviews = (roomId, reviews) => __awaiter(void 0, void 0, void 0, function* () {
    const transaction = yield models_1.default.sequelize.transaction();
    try {
        for (const review of reviews) {
            // Check if review already exists
            const existingReview = yield models_1.default.ExternalReview.findOne({
                where: {
                    source: review.source,
                    externalId: review.id,
                },
                transaction,
            });
            if (!existingReview) {
                // Create new review
                yield models_1.default.ExternalReview.create({
                    roomId,
                    source: review.source,
                    externalId: review.id,
                    rating: review.rating,
                    comment: review.comment,
                    guestName: review.guestName,
                    guestEmail: review.guestEmail,
                    reviewDate: review.reviewDate,
                    isPublished: true,
                }, { transaction });
            }
        }
        yield transaction.commit();
        return true;
    }
    catch (error) {
        yield transaction.rollback();
        console.error('Error updating room reviews:', error);
        throw error;
    }
});
exports.updateRoomReviews = updateRoomReviews;
/**
 * Synchronizes reviews for a specific room or all rooms
 * @param roomId Optional room ID to sync only reviews for this room
 * @param platform Optional platform to sync reviews from (e.g., 'Airbnb', 'Booking.com')
 */
const syncExternalReviews = (roomId, platform) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        // Query to get active external calendars
        const whereClause = { isActive: true };
        if (roomId) {
            whereClause.roomId = roomId;
        }
        if (platform) {
            whereClause.source = platform;
        }
        const externalCalendars = yield models_1.default.ExternalCalendar.findAll({
            where: whereClause,
            include: [{
                    model: models_1.default.Room,
                    as: 'room',
                    attributes: ['id', 'externalIds'],
                }],
        });
        for (const calendar of externalCalendars) {
            try {
                let reviews = [];
                // Fetch reviews based on platform
                switch (calendar.source.toLowerCase()) {
                    case 'airbnb':
                        if ((_a = calendar.room.externalIds) === null || _a === void 0 ? void 0 : _a.airbnb) {
                            reviews = yield (0, exports.fetchAirbnbReviews)(calendar.room.externalIds.airbnb, process.env.AIRBNB_API_KEY || '');
                        }
                        break;
                    case 'booking.com':
                        if ((_b = calendar.room.externalIds) === null || _b === void 0 ? void 0 : _b.booking) {
                            reviews = yield (0, exports.fetchBookingReviews)(calendar.room.externalIds.booking, process.env.BOOKING_API_KEY || '');
                        }
                        break;
                    default:
                        console.warn(`Unsupported platform for review sync: ${calendar.source}`);
                        continue;
                }
                // Update reviews in database
                if (reviews.length > 0) {
                    yield (0, exports.updateRoomReviews)(calendar.roomId, reviews);
                    console.log(`Successfully synced ${reviews.length} reviews from ${calendar.source} for room ${calendar.roomId}`);
                }
            }
            catch (error) {
                console.error(`Failed to sync reviews for room ${calendar.roomId} from ${calendar.source}:`, error);
                // Continue with other rooms/platforms even if one fails
            }
        }
        return true;
    }
    catch (error) {
        console.error('Error syncing external reviews:', error);
        throw error;
    }
});
exports.syncExternalReviews = syncExternalReviews;
