"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateAllRoomsIcs = exports.generateIcs = exports.importReservationsFromIcsContent = exports.parseIcsContent = exports.importReservationsWithCustomDetails = exports.importSelectedReservations = exports.parseAndMapReservations = exports.parseMultiRoomCalendar = void 0;
const axios_1 = __importDefault(require("axios"));
const nodeIcal = __importStar(require("node-ical"));
const models_1 = __importStar(require("../models"));
const sequelize_1 = require("sequelize");
/**
 * Parse iCal content (either from URL or direct content)
 */
const parseMultiRoomCalendarFromContent = (icsContent, source) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    const parsedCal = nodeIcal.parseICS(icsContent);
    const events = [];
    for (const k in parsedCal) {
        const event = parsedCal[k];
        if (event.type === 'VEVENT') {
            // Parse platform-specific data from description
            let reservationUrl = '';
            let phoneNumber = '';
            let guestName = '';
            let guestEmail = '';
            if (event.description) {
                // Airbnb patterns
                const urlMatch = event.description.match(/Reservation URL: (https:\/\/[^\s\n]+)/);
                if (urlMatch)
                    reservationUrl = urlMatch[1];
                const phoneMatch = event.description.match(/Phone Number \(Last 4 Digits\): (\d+)/);
                if (phoneMatch)
                    phoneNumber = phoneMatch[1];
                // Extract guest name from various patterns
                const guestNameMatch = event.description.match(/Guest: ([^\n\r]+)|Name: ([^\n\r]+)|Guest Name: ([^\n\r]+)/i);
                if (guestNameMatch) {
                    guestName = (guestNameMatch[1] || guestNameMatch[2] || guestNameMatch[3]).trim();
                }
                // Extract email if available
                const emailMatch = event.description.match(/([a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/);
                if (emailMatch) {
                    guestEmail = emailMatch[1];
                }
            }
            // Also check summary for guest info (some platforms put it there)
            if (!guestName && event.summary && event.summary !== 'Reserved' && event.summary !== 'Blocked') {
                // Extract potential guest name from summary (remove platform prefixes)
                const cleanSummary = event.summary.replace(/^(Airbnb|Booking\.com|VRBO)\s*[:\-\(]?\s*/i, '').trim();
                if (cleanSummary && cleanSummary.length > 2 && !cleanSummary.match(/^(reserved|blocked|unavailable)$/i)) {
                    guestName = cleanSummary;
                }
            }
            // Determine if this is a reservation or blocked time
            const isReservation = !((_a = event.summary) === null || _a === void 0 ? void 0 : _a.match(/^(blocked|unavailable|maintenance)/i));
            events.push({
                uid: event.uid,
                summary: event.summary || 'Untitled Event',
                description: event.description || '',
                start: event.start,
                end: event.end,
                source,
                calendarId: source === 'uploaded_file' ? 0 : parseInt(source),
                reservationUrl,
                phoneNumber,
                guestName,
                guestEmail,
                isReservation,
                conflictsWith: []
            });
        }
    }
    return events;
});
/**
 * Enhanced parser for multi-room iCal calendars
 * Now extracts guest information and prepares data for user confirmation
 */
const parseMultiRoomCalendar = (url, calendarId) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const response = yield axios_1.default.get(url);
        return yield parseMultiRoomCalendarFromContent(response.data, calendarId.toString());
    }
    catch (error) {
        console.error('Error parsing multi-room calendar:', error);
        throw new Error(`Failed to parse calendar: ${error.message}`);
    }
});
exports.parseMultiRoomCalendar = parseMultiRoomCalendar;
/**
 * Maps events to rooms and detects conflicts - returns data for user review
 */
const parseAndMapReservations = (calendarId) => __awaiter(void 0, void 0, void 0, function* () {
    const calendar = yield models_1.default.MultiRoomCalendar.findByPk(calendarId, {
        include: [
            {
                model: models_1.default.ExternalCalendar,
                as: 'roomMappings',
                where: { isActive: true },
                required: false,
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                        include: [
                            {
                                model: models_1.default.RoomType,
                                as: 'roomType',
                            },
                        ],
                    }
                ]
            }
        ]
    });
    if (!calendar) {
        throw new Error('Calendar not found');
    }
    // Parse events from calendar
    const events = yield (0, exports.parseMultiRoomCalendar)(calendar.calendarUrl, calendarId);
    // Separate reservations from blocked time
    const reservations = events.filter(event => event.isReservation);
    const blockedEvents = events.filter(event => !event.isReservation);
    // Map events to rooms based on existing mappings
    const roomMappings = calendar.roomMappings || [];
    const mappedReservations = [];
    const unmappedEvents = [];
    for (const reservation of reservations) {
        let matchedRoomId = null;
        // Try to match using existing room mappings
        for (const mapping of roomMappings) {
            if (mapping.mappingRules) {
                const rules = Array.isArray(mapping.mappingRules) ? mapping.mappingRules : [mapping.mappingRules];
                for (const rule of rules) {
                    if (matchEvent(reservation, rule)) {
                        matchedRoomId = mapping.roomId;
                        break;
                    }
                }
            }
            if (!matchedRoomId && mapping.externalRoomIdentifier) {
                if (eventMatchesExternalId(reservation, mapping.externalRoomIdentifier)) {
                    matchedRoomId = mapping.roomId;
                }
            }
            if (matchedRoomId)
                break;
        }
        if (matchedRoomId) {
            reservation.mappedRoomId = matchedRoomId;
            mappedReservations.push(reservation);
        }
        else {
            unmappedEvents.push(reservation);
        }
    }
    // Check for conflicts with existing bookings
    const conflicts = [];
    const validReservations = [];
    for (const reservation of mappedReservations) {
        if (reservation.mappedRoomId) {
            const conflictingBookings = yield models_1.default.Booking.findAll({
                where: {
                    roomId: reservation.mappedRoomId,
                    [sequelize_1.Op.and]: [
                        {
                            [sequelize_1.Op.or]: [
                                {
                                    checkInDate: {
                                        [sequelize_1.Op.between]: [
                                            reservation.start.toISOString().split('T')[0],
                                            reservation.end.toISOString().split('T')[0]
                                        ]
                                    }
                                },
                                {
                                    checkOutDate: {
                                        [sequelize_1.Op.between]: [
                                            reservation.start.toISOString().split('T')[0],
                                            reservation.end.toISOString().split('T')[0]
                                        ]
                                    }
                                },
                                {
                                    [sequelize_1.Op.and]: [
                                        {
                                            checkInDate: {
                                                [sequelize_1.Op.lte]: reservation.start.toISOString().split('T')[0]
                                            }
                                        },
                                        {
                                            checkOutDate: {
                                                [sequelize_1.Op.gte]: reservation.end.toISOString().split('T')[0]
                                            }
                                        }
                                    ]
                                }
                            ]
                        }
                    ]
                },
                include: [
                    {
                        model: models_1.default.Guest,
                        as: 'guest'
                    }
                ]
            });
            if (conflictingBookings.length > 0) {
                reservation.conflictsWith = conflictingBookings;
                conflicts.push(reservation);
            }
            else {
                validReservations.push(reservation);
            }
        }
    }
    return {
        events,
        reservations: validReservations,
        conflicts,
        unmappedEvents
    };
});
exports.parseAndMapReservations = parseAndMapReservations;
/**
 * Import selected reservations after user confirmation
 */
const importSelectedReservations = (calendarId_1, reservationUids_1, ...args_1) => __awaiter(void 0, [calendarId_1, reservationUids_1, ...args_1], void 0, function* (calendarId, reservationUids, forceImport = false) {
    const transaction = yield models_1.sequelize.transaction();
    try {
        const results = {
            imported: [],
            failed: [],
            conflicts: []
        };
        // Get the parsed data
        const parsedData = yield (0, exports.parseAndMapReservations)(calendarId);
        const allReservations = [...parsedData.reservations, ...parsedData.conflicts];
        for (const uid of reservationUids) {
            const reservation = allReservations.find(r => r.uid === uid);
            if (!reservation || !reservation.mappedRoomId) {
                results.failed.push({ uid, reason: 'Reservation not found or not mapped' });
                continue;
            }
            // Check for conflicts if not forcing import
            if (!forceImport && reservation.conflictsWith && reservation.conflictsWith.length > 0) {
                results.conflicts.push({
                    uid,
                    reservation,
                    conflicts: reservation.conflictsWith
                });
                continue;
            }
            try {
                // Create or find guest
                const guestEmail = reservation.guestEmail ||
                    `external-${reservation.uid.replace(/[^a-zA-Z0-9]/g, '-').substring(0, 30)}@external.com`;
                const [guest] = yield models_1.default.Guest.findOrCreate({
                    where: { email: guestEmail },
                    defaults: {
                        firstName: reservation.guestName ? reservation.guestName.split(' ')[0] : 'External',
                        lastName: reservation.guestName ? reservation.guestName.split(' ').slice(1).join(' ') || 'Guest' : 'Guest',
                        email: guestEmail,
                        phone: reservation.phoneNumber ? `****${reservation.phoneNumber}` : '+0000000000',
                        address: 'External booking - no address provided',
                        city: 'Unknown',
                        country: 'Unknown'
                    },
                    transaction
                });
                // Get room details for pricing
                const room = yield models_1.default.Room.findByPk(reservation.mappedRoomId, { transaction });
                if (!room) {
                    results.failed.push({ uid, reason: 'Room not found' });
                    continue;
                }
                // Calculate price
                const nights = Math.ceil((reservation.end.getTime() - reservation.start.getTime()) / (1000 * 60 * 60 * 24));
                const totalPrice = nights * room.pricePerNight * (1 - (room.discount || 0) / 100);
                // Create booking
                const booking = yield models_1.default.Booking.create({
                    roomId: reservation.mappedRoomId,
                    guestId: guest.id,
                    checkInDate: reservation.start.toISOString().split('T')[0],
                    checkOutDate: reservation.end.toISOString().split('T')[0],
                    status: 'confirmed',
                    totalPrice,
                    specialRequests: `External booking from ${reservation.source} - UID: ${reservation.uid}${reservation.reservationUrl ? ` - URL: ${reservation.reservationUrl}` : ''}`
                }, { transaction });
                // Update room availability
                const dates = [];
                const currentDate = new Date(reservation.start);
                const endDate = new Date(reservation.end);
                while (currentDate < endDate) {
                    dates.push(new Date(currentDate));
                    currentDate.setDate(currentDate.getDate() + 1);
                }
                // Room availability is now booking-based only - no separate table needed
                results.imported.push({
                    uid,
                    bookingId: booking.id,
                    guestName: reservation.guestName,
                    roomId: reservation.mappedRoomId
                });
            }
            catch (error) {
                results.failed.push({ uid, reason: error.message });
            }
        }
        yield transaction.commit();
        return results;
    }
    catch (error) {
        yield transaction.rollback();
        throw error;
    }
});
exports.importSelectedReservations = importSelectedReservations;
/**
 * Checks if an event matches a specific mapping rule
 */
const matchEvent = (event, rule) => {
    var _a, _b, _c;
    switch (rule.type) {
        case 'url_contains':
            return ((_a = event.reservationUrl) === null || _a === void 0 ? void 0 : _a.includes(rule.value)) || false;
        case 'uid_contains':
            return event.uid.includes(rule.value);
        case 'description_contains':
            return ((_b = event.description) === null || _b === void 0 ? void 0 : _b.includes(rule.value)) || false;
        case 'phone_pattern':
            return ((_c = event.phoneNumber) === null || _c === void 0 ? void 0 : _c.includes(rule.value)) || false;
        default:
            return false;
    }
};
/**
 * Import reservations with custom guest details and room mappings
 */
const importReservationsWithCustomDetails = (calendarId_1, reservationsData_1, ...args_1) => __awaiter(void 0, [calendarId_1, reservationsData_1, ...args_1], void 0, function* (calendarId, reservationsData, forceImport = false) {
    var _a, _b;
    const transaction = yield models_1.sequelize.transaction();
    try {
        const results = {
            imported: [],
            failed: [],
            conflicts: []
        };
        // Get the parsed data from ICS to match with UIDs
        const parsedData = yield (0, exports.parseAndMapReservations)(calendarId);
        const allReservations = [...parsedData.reservations, ...parsedData.conflicts, ...parsedData.unmappedEvents];
        for (const reservationData of reservationsData) {
            const { uid, roomId, guestDetails, summary, start, end } = reservationData;
            // Find the original reservation from ICS
            const originalReservation = allReservations.find(r => r.uid === uid);
            if (!originalReservation) {
                results.failed.push({ uid, reason: 'Reservation not found in ICS calendar' });
                continue;
            }
            if (!roomId) {
                results.failed.push({ uid, reason: 'Room ID is required' });
                continue;
            }
            // Check for existing booking with same UID (prevent duplicates)
            const existingBooking = yield models_1.default.Booking.findOne({
                where: { externalId: uid }
            });
            if (existingBooking && !forceImport) {
                results.conflicts.push({ uid, reason: 'Booking already exists', existingId: existingBooking.id });
                continue;
            }
            // Check for date conflicts in the same room
            const checkInDate = new Date(start || originalReservation.start);
            const checkOutDate = new Date(end || originalReservation.end);
            const conflictingBooking = yield models_1.default.Booking.findOne({
                where: {
                    roomId: roomId,
                    [sequelize_1.Op.or]: [
                        {
                            checkInDate: {
                                [sequelize_1.Op.between]: [checkInDate, checkOutDate]
                            }
                        },
                        {
                            checkOutDate: {
                                [sequelize_1.Op.between]: [checkInDate, checkOutDate]
                            }
                        },
                        {
                            [sequelize_1.Op.and]: [
                                { checkInDate: { [sequelize_1.Op.lte]: checkInDate } },
                                { checkOutDate: { [sequelize_1.Op.gte]: checkOutDate } }
                            ]
                        }
                    ]
                }
            });
            if (conflictingBooking && !forceImport) {
                results.conflicts.push({
                    uid,
                    reason: 'Date conflict with existing booking',
                    conflictingBookingId: conflictingBooking.id,
                    conflictDates: `${conflictingBooking.checkInDate} - ${conflictingBooking.checkOutDate}`
                });
                continue;
            }
            try {
                // Create or find guest
                let guest = null;
                if (guestDetails && guestDetails.email) {
                    // Try to find existing guest by email
                    guest = yield models_1.default.Guest.findOne({
                        where: { email: guestDetails.email }
                    });
                    if (!guest) {
                        // Create new guest
                        guest = yield models_1.default.Guest.create({
                            firstName: guestDetails.firstName || 'External',
                            lastName: guestDetails.lastName || 'Guest',
                            email: guestDetails.email,
                            phone: guestDetails.phone || '',
                            address: guestDetails.address || '',
                            city: guestDetails.city || '',
                            country: guestDetails.country || ''
                        }, { transaction });
                    }
                }
                else {
                    // Create default guest from original reservation data
                    const defaultGuestData = {
                        firstName: ((_a = originalReservation.guestName) === null || _a === void 0 ? void 0 : _a.split(' ')[0]) || 'External',
                        lastName: ((_b = originalReservation.guestName) === null || _b === void 0 ? void 0 : _b.split(' ').slice(1).join(' ')) || 'Guest',
                        email: originalReservation.guestEmail || `external.${uid}@placeholder.com`,
                        phone: originalReservation.phoneNumber || '',
                        address: 'External booking - no address provided',
                        city: 'Unknown',
                        country: 'Unknown'
                    };
                    guest = yield models_1.default.Guest.create(defaultGuestData, { transaction });
                }
                // Calculate total price (basic calculation)
                const nights = Math.ceil((checkOutDate.getTime() - checkInDate.getTime()) / (1000 * 60 * 60 * 24));
                const room = yield models_1.default.Room.findByPk(roomId);
                const totalAmount = room ? room.pricePerNight * nights : 0;
                // Create booking
                const booking = yield models_1.default.Booking.create({
                    guestId: guest.id,
                    roomId: roomId,
                    checkInDate: checkInDate,
                    checkOutDate: checkOutDate,
                    totalPrice: totalAmount,
                    status: 'confirmed',
                    paymentStatus: 'pending',
                    source: originalReservation.source || 'external',
                    externalId: uid,
                    notes: `Imported from ${originalReservation.source || 'external calendar'}: ${summary || originalReservation.summary}`,
                    createdAt: new Date(),
                    updatedAt: new Date()
                }, { transaction });
                results.imported.push({
                    uid,
                    bookingId: booking.id,
                    guestId: guest.id,
                    roomId: roomId,
                    checkInDate,
                    checkOutDate,
                    totalAmount
                });
            }
            catch (error) {
                console.error(`Failed to create booking for UID ${uid}:`, error);
                results.failed.push({
                    uid,
                    reason: `Database error: ${error.message}`,
                    details: error.message
                });
            }
        }
        yield transaction.commit();
        return results;
    }
    catch (error) {
        yield transaction.rollback();
        throw error;
    }
});
exports.importReservationsWithCustomDetails = importReservationsWithCustomDetails;
/**
 * Parse ICS content from uploaded file (reuses existing parsing logic)
 */
const parseIcsContent = (icsContent_1, ...args_1) => __awaiter(void 0, [icsContent_1, ...args_1], void 0, function* (icsContent, roomMappings = []) {
    // Reuse existing parsing logic but with content instead of URL
    const events = yield parseMultiRoomCalendarFromContent(icsContent, 'uploaded_file');
    // Apply room mappings if provided
    const mappedReservations = [];
    const unmappedEvents = [];
    for (const event of events) {
        let matched = false;
        for (const mapping of roomMappings) {
            if (mapping.externalRoomIdentifier && eventMatchesExternalId(event, mapping.externalRoomIdentifier)) {
                event.mappedRoomId = mapping.roomId;
                mappedReservations.push(event);
                matched = true;
                break;
            }
        }
        if (!matched) {
            unmappedEvents.push(event);
        }
    }
    return {
        reservations: mappedReservations,
        conflicts: [], // File uploads don't have conflicts with other calendars
        unmappedEvents: unmappedEvents
    };
});
exports.parseIcsContent = parseIcsContent;
/**
 * Import reservations from ICS file content
 */
const importReservationsFromIcsContent = (icsContent_1, reservationsData_1, ...args_1) => __awaiter(void 0, [icsContent_1, reservationsData_1, ...args_1], void 0, function* (icsContent, reservationsData, roomMappings = [], forceImport = false) {
    var _a, _b;
    const transaction = yield models_1.sequelize.transaction();
    try {
        const results = {
            imported: [],
            failed: [],
            conflicts: []
        };
        // Parse the ICS content to get the original reservations
        const parsedData = yield (0, exports.parseIcsContent)(icsContent, roomMappings);
        const allReservations = [...parsedData.reservations, ...parsedData.unmappedEvents];
        for (const reservationData of reservationsData) {
            const { uid, roomId, guestDetails, summary, start, end } = reservationData;
            // Find the original reservation from ICS
            const originalReservation = allReservations.find(r => r.uid === uid);
            if (!originalReservation) {
                results.failed.push({ uid, reason: 'Reservation not found in ICS content' });
                continue;
            }
            if (!roomId) {
                results.failed.push({ uid, reason: 'Room ID is required' });
                continue;
            }
            // Check for existing booking with same UID (prevent duplicates)
            const existingBooking = yield models_1.default.Booking.findOne({
                where: { externalId: uid }
            });
            if (existingBooking && !forceImport) {
                results.conflicts.push({ uid, reason: 'Booking already exists', existingId: existingBooking.id });
                continue;
            }
            // Check for date conflicts in the same room
            const checkInDate = new Date(start || originalReservation.start);
            const checkOutDate = new Date(end || originalReservation.end);
            const conflictingBooking = yield models_1.default.Booking.findOne({
                where: {
                    roomId: roomId,
                    [sequelize_1.Op.or]: [
                        {
                            checkInDate: {
                                [sequelize_1.Op.between]: [checkInDate, checkOutDate]
                            }
                        },
                        {
                            checkOutDate: {
                                [sequelize_1.Op.between]: [checkInDate, checkOutDate]
                            }
                        },
                        {
                            [sequelize_1.Op.and]: [
                                { checkInDate: { [sequelize_1.Op.lte]: checkInDate } },
                                { checkOutDate: { [sequelize_1.Op.gte]: checkOutDate } }
                            ]
                        }
                    ]
                }
            });
            if (conflictingBooking && !forceImport) {
                results.conflicts.push({
                    uid,
                    reason: 'Date conflict with existing booking',
                    conflictingBookingId: conflictingBooking.id,
                    conflictDates: `${conflictingBooking.checkInDate} - ${conflictingBooking.checkOutDate}`
                });
                continue;
            }
            try {
                // Create or find guest
                let guest = null;
                if (guestDetails && guestDetails.email) {
                    // Try to find existing guest by email
                    guest = yield models_1.default.Guest.findOne({
                        where: { email: guestDetails.email }
                    });
                    if (!guest) {
                        // Create new guest
                        guest = yield models_1.default.Guest.create({
                            firstName: guestDetails.firstName || 'External',
                            lastName: guestDetails.lastName || 'Guest',
                            email: guestDetails.email,
                            phone: guestDetails.phone || '',
                            address: guestDetails.address || '',
                            city: guestDetails.city || '',
                            country: guestDetails.country || ''
                        }, { transaction });
                    }
                }
                else {
                    // Create default guest from original reservation data
                    const defaultGuestData = {
                        firstName: ((_a = originalReservation.guestName) === null || _a === void 0 ? void 0 : _a.split(' ')[0]) || 'External',
                        lastName: ((_b = originalReservation.guestName) === null || _b === void 0 ? void 0 : _b.split(' ').slice(1).join(' ')) || 'Guest',
                        email: originalReservation.guestEmail || `external.${uid}@placeholder.com`,
                        phone: originalReservation.phoneNumber || '',
                        address: 'External booking - no address provided',
                        city: 'Unknown',
                        country: 'Unknown'
                    };
                    guest = yield models_1.default.Guest.create(defaultGuestData, { transaction });
                }
                // Calculate total price (basic calculation)
                const nights = Math.ceil((checkOutDate.getTime() - checkInDate.getTime()) / (1000 * 60 * 60 * 24));
                const room = yield models_1.default.Room.findByPk(roomId);
                const totalAmount = room ? room.pricePerNight * nights : 0;
                // Create booking
                const booking = yield models_1.default.Booking.create({
                    guestId: guest.id,
                    roomId: roomId,
                    checkInDate: checkInDate,
                    checkOutDate: checkOutDate,
                    totalPrice: totalAmount,
                    status: 'confirmed',
                    externalId: uid,
                    specialRequests: `Imported from uploaded ICS file: ${summary || originalReservation.summary}`,
                    createdAt: new Date(),
                    updatedAt: new Date()
                }, { transaction });
                results.imported.push({
                    uid,
                    bookingId: booking.id,
                    guestId: guest.id,
                    roomId: roomId,
                    checkInDate,
                    checkOutDate,
                    totalAmount
                });
            }
            catch (error) {
                console.error(`Failed to create booking for UID ${uid}:`, error);
                results.failed.push({
                    uid,
                    reason: `Database error: ${error.message}`,
                    details: error.message
                });
            }
        }
        yield transaction.commit();
        return results;
    }
    catch (error) {
        yield transaction.rollback();
        throw error;
    }
});
exports.importReservationsFromIcsContent = importReservationsFromIcsContent;
/**
 * Checks if event matches external room identifier
 */
const eventMatchesExternalId = (event, externalId) => {
    var _a, _b;
    return event.uid.includes(externalId) ||
        ((_a = event.reservationUrl) === null || _a === void 0 ? void 0 : _a.includes(externalId)) ||
        ((_b = event.description) === null || _b === void 0 ? void 0 : _b.includes(externalId)) || false;
};
/**
 * Generates an iCalendar (.ics) file content for a given room's bookings.
 * @param roomId The ID of the room to export.
 * @returns A string containing the iCalendar data.
 */
const generateIcs = (roomId) => __awaiter(void 0, void 0, void 0, function* () {
    const room = yield models_1.default.Room.findByPk(roomId, {
        include: [{
                model: models_1.default.Booking,
                as: 'bookings',
                where: {
                    status: { [sequelize_1.Op.in]: ['confirmed', 'checked_in', 'paid'] }
                },
                required: false,
                include: [{
                        model: models_1.default.Guest,
                        as: 'guest'
                    }]
            }]
    });
    if (!room) {
        throw new Error('Room not found');
    }
    const formatDateForIcs = (date) => {
        return date.toISOString().replace(/[-:]/g, '').split('.')[0] + 'Z';
    };
    const cal = [
        'BEGIN:VCALENDAR',
        'VERSION:2.0',
        `PRODID:-//CasaDelMar//Room ${room.id} Calendar//EN`,
        'CALSCALE:GREGORIAN',
        'METHOD:PUBLISH',
        `X-WR-CALNAME:Casa del Mar - ${room.name}`,
        'X-WR-TIMEZONE:UTC',
    ];
    if (room.bookings) {
        for (const booking of room.bookings) {
            const start = new Date(booking.checkInDate);
            const end = new Date(booking.checkOutDate);
            // ICS vevents are exclusive of the end date for all-day events.
            // For timed events, it's inclusive. We'll treat them as timed events
            // to be safe, from standard check-in to check-out times.
            if (start.getUTCHours() === 0)
                start.setUTCHours(15); // 3 PM UTC
            if (end.getUTCHours() === 0)
                end.setUTCHours(11); // 11 AM UTC
            const guestName = booking.guest ? `${booking.guest.firstName} ${booking.guest.lastName}`.trim() : 'Guest';
            cal.push('BEGIN:VEVENT');
            // Use booking UUID for our own bookings, or external ID if it exists
            cal.push(`UID:${booking.externalId || booking.id}@casadelmar.com`);
            cal.push(`DTSTAMP:${formatDateForIcs(new Date())}`);
            cal.push(`DTSTART:${formatDateForIcs(start)}`);
            cal.push(`DTEND:${formatDateForIcs(end)}`);
            cal.push(`SUMMARY:Reserved - ${guestName === 'Guest' ? 'Blocked' : guestName}`);
            cal.push(`DESCRIPTION:Booking for ${guestName}. Status: ${booking.status}. Booking ID: ${booking.id}`);
            cal.push('END:VEVENT');
        }
    }
    cal.push('END:VCALENDAR');
    return cal.join('\r\n');
});
exports.generateIcs = generateIcs;
/**
 * Generates an iCalendar (.ics) file content for all rooms and their bookings.
 * @returns A string containing the iCalendar data for all rooms.
 */
const generateAllRoomsIcs = () => __awaiter(void 0, void 0, void 0, function* () {
    const rooms = yield models_1.default.Room.findAll({
        include: [{
                model: models_1.default.Booking,
                as: 'bookings',
                where: {
                    status: { [sequelize_1.Op.in]: ['confirmed', 'checked_in', 'paid'] }
                },
                required: false,
                include: [{
                        model: models_1.default.Guest,
                        as: 'guest'
                    }, {
                        model: models_1.default.Payment,
                        as: 'payments',
                        required: false
                    }]
            }]
    });
    const formatDateForIcs = (date) => {
        return date.toISOString().replace(/[-:]/g, '').split('.')[0] + 'Z';
    };
    const cal = [
        'BEGIN:VCALENDAR',
        'VERSION:2.0',
        `PRODID:-//CasaDelMar//All Rooms Calendar//EN`,
        'CALSCALE:GREGORIAN',
        'METHOD:PUBLISH',
        `X-WR-CALNAME:Casa del Mar - All Rooms`,
        'X-WR-TIMEZONE:UTC',
    ];
    for (const room of rooms) {
        if (room.bookings) {
            for (const booking of room.bookings) {
                const start = new Date(booking.checkInDate);
                const end = new Date(booking.checkOutDate);
                if (start.getUTCHours() === 0)
                    start.setUTCHours(15); // 3 PM UTC
                if (end.getUTCHours() === 0)
                    end.setUTCHours(11); // 11 AM UTC
                const guestName = booking.guest ? `${booking.guest.firstName} ${booking.guest.lastName}`.trim() : 'Guest';
                // Determine payment status from associated payments (hasMany: 'payments')
                let paymentStatus = 'unpaid';
                if (Array.isArray(booking.payments) && booking.payments.length > 0) {
                    if (booking.payments.some((p) => p.status === 'paid')) {
                        paymentStatus = 'paid';
                    }
                    else if (booking.payments.some((p) => p.status === 'failed')) {
                        paymentStatus = 'failed';
                    }
                    else if (booking.payments.some((p) => p.status === 'refunded')) {
                        paymentStatus = 'refunded';
                    }
                    else if (booking.payments.some((p) => p.status === 'pending')) {
                        paymentStatus = 'pending';
                    }
                }
                cal.push('BEGIN:VEVENT');
                cal.push(`UID:${booking.externalId || booking.id}@casadelmar.com`);
                cal.push(`DTSTAMP:${formatDateForIcs(new Date())}`);
                cal.push(`DTSTART:${formatDateForIcs(start)}`);
                cal.push(`DTEND:${formatDateForIcs(end)}`);
                cal.push(`SUMMARY:${room.name} - Reserved - ${guestName === 'Guest' ? 'Blocked' : guestName}`);
                cal.push(`DESCRIPTION:Booking for ${guestName}. Status: ${booking.status}. Payment: ${paymentStatus}. Booking ID: ${booking.id}`);
                cal.push('END:VEVENT');
            }
        }
    }
    cal.push('END:VCALENDAR');
    return cal.join('\r\n');
});
exports.generateAllRoomsIcs = generateAllRoomsIcs;
