"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateRoomCalendarFeed = exports.syncExternalCalendars = exports.updateRoomAvailability = exports.fetchCalendar = void 0;
const axios_1 = __importDefault(require("axios"));
const nodeIcal = __importStar(require("node-ical"));
const models_1 = __importDefault(require("../models"));
const sequelize_1 = require("sequelize");
/**
 * Fetches and parses an iCal calendar from the provided URL
 * @param url The URL of the iCal calendar
 * @returns Array of calendar events
 */
const fetchCalendar = (url) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const response = yield axios_1.default.get(url);
        const parsedCal = nodeIcal.parseICS(response.data);
        const events = [];
        for (const k in parsedCal) {
            const event = parsedCal[k];
            if (event.type === 'VEVENT') {
                events.push({
                    uid: event.uid,
                    summary: event.summary,
                    start: event.start,
                    end: event.end,
                    source: url
                });
            }
        }
        return events;
    }
    catch (error) {
        console.error('Error fetching or parsing calendar:', error);
        throw new Error(`Failed to fetch or parse calendar: ${error.message}`);
    }
});
exports.fetchCalendar = fetchCalendar;
/**
 * Updates room availability based on external calendar events
 * @param roomId The ID of the room
 * @param events Array of external calendar events
 */
const updateRoomAvailability = (roomId, events) => __awaiter(void 0, void 0, void 0, function* () {
    const transaction = yield models_1.default.sequelize.transaction();
    try {
        // Get all date ranges from events
        for (const event of events) {
            const start = new Date(event.start);
            const end = new Date(event.end);
            // Create a date range between start and end (exclusive of end date)
            const dates = [];
            const currentDate = new Date(start);
            while (currentDate < end) {
                dates.push(new Date(currentDate));
                currentDate.setDate(currentDate.getDate() + 1);
            }
            // For each date in the range, mark the room as unavailable
            for (const date of dates) {
                const formattedDate = date.toISOString().split('T')[0]; // YYYY-MM-DD format
                // Room availability is now booking-based only - external blocks should be handled as special bookings
                console.log(`External event blocking room ${roomId} on ${currentDate.toISOString().split('T')[0]} - consider creating maintenance booking`);
            }
        }
        yield transaction.commit();
        return true;
    }
    catch (error) {
        yield transaction.rollback();
        console.error('Error updating room availability:', error);
        throw error;
    }
});
exports.updateRoomAvailability = updateRoomAvailability;
/**
 * Synchronizes all multi-room calendars for a specific room or all rooms
 * @param roomId Optional room ID to sync only calendars for this room
 */
const syncExternalCalendars = (roomId) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Build query to get active multi-room calendars with their room mappings
        const includeOptions = {
            model: models_1.default.ExternalCalendar,
            as: 'roomMappings',
            where: { isActive: true },
            include: [{
                    model: models_1.default.Room,
                    as: 'room'
                }]
        };
        // If specific room requested, filter mappings
        if (roomId) {
            includeOptions.where.roomId = roomId;
        }
        const multiRoomCalendars = yield models_1.default.MultiRoomCalendar.findAll({
            where: { isActive: true },
            include: [includeOptions]
        });
        for (const calendar of multiRoomCalendars) {
            try {
                // Fetch calendar events once per ICS link
                const events = yield (0, exports.fetchCalendar)(calendar.calendarUrl);
                // Process events for each mapped room
                for (const mapping of calendar.roomMappings) {
                    try {
                        // Filter events for this specific room based on mapping rules
                        const roomEvents = filterEventsForRoom(events, mapping);
                        yield (0, exports.updateRoomAvailability)(mapping.roomId, roomEvents);
                        console.log(`Successfully synced room ${mapping.roomId} from calendar: ${calendar.name} (${calendar.source})`);
                    }
                    catch (error) {
                        console.error(`Failed to sync room ${mapping.roomId} from calendar ${calendar.id}:`, error);
                        // Continue with other rooms even if one fails
                    }
                }
                // Update last synced timestamp for the calendar
                yield calendar.update({
                    lastSynced: new Date()
                });
                console.log(`Successfully synced multi-room calendar: ${calendar.name} (${calendar.source})`);
            }
            catch (error) {
                console.error(`Failed to sync multi-room calendar ${calendar.id}:`, error);
                // Continue with other calendars even if one fails
            }
        }
        return true;
    }
    catch (error) {
        console.error('Error syncing external calendars:', error);
        throw error;
    }
});
exports.syncExternalCalendars = syncExternalCalendars;
/**
 * Filters events for a specific room based on mapping rules
 * @param events All events from the calendar
 * @param mapping Room calendar mapping with filtering rules
 * @returns Events filtered for the specific room
 */
const filterEventsForRoom = (events, mapping) => {
    if (!mapping.mappingRules) {
        // If no mapping rules, return all events (fallback behavior)
        return events;
    }
    const rules = mapping.mappingRules;
    return events.filter(event => {
        // Filter by external room identifier if specified
        if (rules.externalRoomId && mapping.externalRoomIdentifier) {
            // Check if event summary or description contains the external room identifier
            const eventText = `${event.summary} ${event.uid}`.toLowerCase();
            const roomId = mapping.externalRoomIdentifier.toLowerCase();
            if (!eventText.includes(roomId)) {
                return false;
            }
        }
        // Filter by keywords if specified
        if (rules.includeKeywords && rules.includeKeywords.length > 0) {
            const eventText = `${event.summary}`.toLowerCase();
            const hasKeyword = rules.includeKeywords.some((keyword) => eventText.includes(keyword.toLowerCase()));
            if (!hasKeyword) {
                return false;
            }
        }
        // Exclude by keywords if specified
        if (rules.excludeKeywords && rules.excludeKeywords.length > 0) {
            const eventText = `${event.summary}`.toLowerCase();
            const hasExcludeKeyword = rules.excludeKeywords.some((keyword) => eventText.includes(keyword.toLowerCase()));
            if (hasExcludeKeyword) {
                return false;
            }
        }
        return true;
    });
};
/**
 * Exports the room's availability as an iCal feed
 * @param roomId The ID of the room
 * @returns iCal formatted string
 */
const generateRoomCalendarFeed = (roomId) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const icalGenerator = require('ical-generator').default;
        // Get room details
        const room = yield models_1.default.Room.findByPk(roomId, {
            include: [{ model: models_1.default.RoomType, as: 'roomType' }]
        });
        if (!room) {
            throw new Error('Room not found');
        }
        // Get all bookings for this room
        const bookings = yield models_1.default.Booking.findAll({
            where: {
                roomId,
                status: {
                    [sequelize_1.Op.in]: ['confirmed', 'pending'] // Only include confirmed and pending bookings
                }
            },
            include: [{ model: models_1.default.Guest, as: 'guest' }]
        });
        // Create calendar
        const cal = icalGenerator({
            name: `${room.number} - ${room.roomType.name}`,
            prodId: { company: 'CasaDelMar', product: 'Booking System' }
        });
        // Add each booking as an event
        bookings.forEach(booking => {
            cal.createEvent({
                uid: `booking-${booking.id}`,
                start: booking.checkInDate,
                end: booking.checkOutDate,
                summary: `Room ${room.number} Booked`,
                description: `Booked by ${booking.guest.firstName} ${booking.guest.lastName}`,
                status: booking.status === 'confirmed' ? 'confirmed' : 'tentative'
            });
        });
        return cal.toString();
    }
    catch (error) {
        console.error('Error generating room calendar feed:', error);
        throw error;
    }
});
exports.generateRoomCalendarFeed = generateRoomCalendarFeed;
