"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateGuestDetails = exports.extractGuestDetailsFromEvents = exports.extractGuestDetailsFromSummary = void 0;
/**
 * Extracts guest details from calendar event summary
 * @param summary - The event summary/title from calendar
 * @returns Partial guest details extracted from the summary
 */
const extractGuestDetailsFromSummary = (summary) => {
    if (!summary)
        return {};
    const details = {};
    // Clean up the summary - remove common booking platform indicators
    const cleanSummary = summary.replace(/\b(reserved|booking|reservation|airbnb|vrbo|blocked|unavailable)\b/gi, '').trim();
    // Try to extract name from summary (common patterns)
    const nameParts = cleanSummary.split(/[\s\-,]+/).filter(part => part.length > 1 &&
        !['by', 'for', 'guest', 'room', 'night', 'nights', 'check', 'in', 'out'].includes(part.toLowerCase()) &&
        !/^\d+$/.test(part) // Exclude pure numbers
    );
    if (nameParts.length >= 1) {
        details.firstName = nameParts[0];
    }
    if (nameParts.length >= 2) {
        details.lastName = nameParts.slice(1).join(' ');
    }
    // Look for email pattern in summary
    const emailMatch = summary.match(/[\w.-]+@[\w.-]+\.\w+/);
    if (emailMatch) {
        details.email = emailMatch[0];
    }
    // Look for phone pattern in summary (various formats)
    const phonePatterns = [
        /\+?1?[-.\s]?\(?(\d{3})\)?[-.\s]?(\d{3})[-.\s]?(\d{4})/, // US format
        /\+?(\d{1,3})[-.\s]?(\d{3,4})[-.\s]?(\d{3,4})[-.\s]?(\d{3,4})/, // International
        /[\+]?[\d\s\-\(\)]{10,}/ // General pattern
    ];
    for (const pattern of phonePatterns) {
        const phoneMatch = summary.match(pattern);
        if (phoneMatch) {
            details.phone = phoneMatch[0].replace(/[^\d\+]/g, '');
            break;
        }
    }
    return details;
};
exports.extractGuestDetailsFromSummary = extractGuestDetailsFromSummary;
/**
 * Extracts guest details from multiple calendar events
 * @param events - Array of calendar events with summaries
 * @returns Array of events with extracted guest details
 */
const extractGuestDetailsFromEvents = (events) => {
    return events.map(event => (Object.assign(Object.assign({}, event), { extractedGuestDetails: (0, exports.extractGuestDetailsFromSummary)(event.summary || event.title || '') })));
};
exports.extractGuestDetailsFromEvents = extractGuestDetailsFromEvents;
/**
 * Validates extracted guest details
 * @param details - Guest details to validate
 * @returns Validation result with errors
 */
const validateGuestDetails = (details) => {
    const errors = [];
    if (!details.firstName || details.firstName.length < 1) {
        errors.push('First name is required');
    }
    if (details.email && !/^[\w.-]+@[\w.-]+\.\w+$/.test(details.email)) {
        errors.push('Invalid email format');
    }
    if (details.phone && !/^[\+]?[\d\s\-\(\)]{10,}$/.test(details.phone)) {
        errors.push('Invalid phone format');
    }
    return {
        isValid: errors.length === 0,
        errors
    };
};
exports.validateGuestDetails = validateGuestDetails;
exports.default = {
    extractGuestDetailsFromSummary: exports.extractGuestDetailsFromSummary,
    extractGuestDetailsFromEvents: exports.extractGuestDetailsFromEvents,
    validateGuestDetails: exports.validateGuestDetails
};
