"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFilename = exports.getRelativePath = void 0;
const multer_1 = __importDefault(require("multer"));
const path_1 = __importDefault(require("path"));
const uuid_1 = require("uuid");
const fs_1 = __importDefault(require("fs"));
// Define the storage strategy for multer
const storage = multer_1.default.diskStorage({
    destination: (req, file, cb) => {
        // Determine the destination path based on file type
        let uploadPath = '';
        if (file.fieldname === 'roomImages') {
            uploadPath = path_1.default.join(__dirname, '../public/uploads/rooms');
        }
        else {
            uploadPath = path_1.default.join(__dirname, '../public/uploads/misc');
        }
        // Create directory if it doesn't exist
        if (!fs_1.default.existsSync(uploadPath)) {
            fs_1.default.mkdirSync(uploadPath, { recursive: true });
        }
        cb(null, uploadPath);
    },
    filename: (req, file, cb) => {
        // Create unique filename with original extension
        const uniqueFilename = `${(0, uuid_1.v4)()}${path_1.default.extname(file.originalname)}`;
        cb(null, uniqueFilename);
    }
});
// File filter to allow only images
const fileFilter = (req, file, cb) => {
    // Accept only image files
    if (file.mimetype.startsWith('image/')) {
        cb(null, true);
    }
    else {
        cb(new Error('Only image files are allowed!'));
    }
};
// Create the multer instance
const upload = (0, multer_1.default)({
    storage,
    fileFilter,
    limits: {
        fileSize: 10 * 1024 * 1024, // 10MB max file size
    }
});
// Helper function to get the path relative to public directory
const getRelativePath = (absolutePath) => {
    const publicDir = path_1.default.join(__dirname, '../public');
    return absolutePath.replace(publicDir, '');
};
exports.getRelativePath = getRelativePath;
// Helper function to get just the filename from a path
const getFilename = (absolutePath) => {
    return path_1.default.basename(absolutePath);
};
exports.getFilename = getFilename;
exports.default = upload;
