"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.catchAsync = exports.notFoundHandler = exports.globalErrorHandler = exports.AppError = void 0;
// Custom error class
class AppError extends Error {
    constructor(message, statusCode) {
        super(message);
        this.statusCode = statusCode;
        this.isOperational = true;
        Error.captureStackTrace(this, this.constructor);
    }
}
exports.AppError = AppError;
// Development error response
const sendErrorDev = (err, res) => {
    return res.status(err.statusCode).json({
        status: 'error',
        message: err.message,
        error: err,
        stack: err.stack,
    });
};
// Production error response
const sendErrorProd = (err, res) => {
    // Operational, trusted error: send message to client
    if (err.isOperational) {
        return res.status(err.statusCode).json({
            status: 'error',
            message: err.message,
        });
    }
    // Programming or other unknown error: don't leak error details
    console.error('ERROR 💥', err);
    return res.status(500).json({
        status: 'error',
        message: 'Something went wrong',
    });
};
// Global error handler middleware
const globalErrorHandler = (err, req, res, next) => {
    const error = err;
    error.statusCode = error.statusCode || 500;
    // Comprehensive error logging
    console.log('\n🚨 ========== ERROR OCCURRED ==========');
    console.log('📅 Timestamp:', new Date().toISOString());
    console.log('🔗 Request URL:', `${req.method} ${req.originalUrl}`);
    console.log('📍 Request IP:', req.ip || req.connection.remoteAddress);
    console.log('👤 User Agent:', req.get('User-Agent') || 'Unknown');
    if (req.body && Object.keys(req.body).length > 0) {
        console.log('📦 Request Body:', JSON.stringify(req.body, null, 2));
    }
    if (req.params && Object.keys(req.params).length > 0) {
        console.log('🎯 Request Params:', JSON.stringify(req.params, null, 2));
    }
    if (req.query && Object.keys(req.query).length > 0) {
        console.log('❓ Request Query:', JSON.stringify(req.query, null, 2));
    }
    console.log('\n💥 ERROR DETAILS:');
    console.log('📝 Error Name:', error.name || 'Unknown Error');
    console.log('💬 Error Message:', error.message || 'No message provided');
    console.log('🔢 Status Code:', error.statusCode || 500);
    console.log('🏷️  Is Operational:', error.isOperational || false);
    if (error.stack) {
        console.log('\n📚 Stack Trace:');
        console.log(error.stack);
    }
    // Error type explanation
    console.log('\n💡 ERROR EXPLANATION:');
    if (error.name === 'CastError') {
        console.log('🔍 Type: Database Cast Error - Invalid data type provided for database field');
    }
    else if (error.name === 'ValidationError') {
        console.log('🔍 Type: Validation Error - Data validation failed');
    }
    else if (error.name === 'JsonWebTokenError') {
        console.log('🔍 Type: JWT Error - Invalid or malformed authentication token');
    }
    else if (error.name === 'TokenExpiredError') {
        console.log('🔍 Type: Token Expired - Authentication token has expired');
    }
    else if (error.name === 'SequelizeValidationError') {
        console.log('🔍 Type: Database Validation Error - Database model validation failed');
    }
    else if (error.name === 'SequelizeUniqueConstraintError') {
        console.log('🔍 Type: Database Unique Constraint Error - Duplicate value for unique field');
    }
    else if (error.name === 'MulterError') {
        console.log('🔍 Type: File Upload Error - Issue with file upload processing');
    }
    else if (error.statusCode === 404) {
        console.log('🔍 Type: Not Found Error - Requested resource does not exist');
    }
    else if (error.statusCode === 403) {
        console.log('🔍 Type: Forbidden Error - User lacks permission for this action');
    }
    else if (error.statusCode === 401) {
        console.log('🔍 Type: Unauthorized Error - Authentication required or failed');
    }
    else if (error.statusCode >= 500) {
        console.log('🔍 Type: Server Error - Internal application error occurred');
    }
    else {
        console.log('🔍 Type: Custom Application Error - Business logic error');
    }
    console.log('========================================\n');
    if (process.env.NODE_ENV === 'development') {
        sendErrorDev(error, res);
    }
    else {
        let errorObj = Object.assign({}, error);
        if (error.name === 'CastError') {
            errorObj = new AppError(`Invalid ${error.path}: ${error.value}`, 400);
        }
        if (error.name === 'ValidationError') {
            const errors = Object.values(error.errors).map((el) => el.message);
            errorObj = new AppError(`Invalid input data. ${errors.join('. ')}`, 400);
        }
        if (error.name === 'JsonWebTokenError') {
            errorObj = new AppError('Invalid token. Please log in again.', 401);
        }
        if (error.name === 'TokenExpiredError') {
            errorObj = new AppError('Your token has expired. Please log in again.', 401);
        }
        sendErrorProd(errorObj, res);
    }
};
exports.globalErrorHandler = globalErrorHandler;
// Not found error handler
const notFoundHandler = (req, res, next) => {
    const error = new AppError(`Cannot find ${req.originalUrl} on this server!`, 404);
    next(error);
};
exports.notFoundHandler = notFoundHandler;
// Async error handler
const catchAsync = (fn) => {
    return (req, res, next) => {
        fn(req, res, next).catch(next);
    };
};
exports.catchAsync = catchAsync;
