"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PaymentAccessTokenService = void 0;
const crypto_1 = __importDefault(require("crypto"));
const sequelize_1 = require("sequelize");
const models_1 = __importDefault(require("../models"));
const logger_1 = __importDefault(require("../utils/logger"));
/**
 * Payment Access Token Service
 * Manages temporary access tokens for payment result pages
 */
class PaymentAccessTokenService {
    /**
     * Generate a secure access token for payment results
     */
    static generateAccessToken() {
        return crypto_1.default.randomBytes(32).toString('hex');
    }
    /**
     * Create a new payment access token
     */
    static createAccessToken(paymentId, bookingId, ipAddress, userAgent) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const token = this.generateAccessToken();
                const expiresAt = new Date(Date.now() + 24 * 60 * 60 * 1000); // 24 hours from now
                yield models_1.default.PaymentAccessToken.create({
                    token,
                    paymentId,
                    bookingId,
                    ipAddress,
                    userAgent,
                    expiresAt,
                    isUsed: false
                });
                logger_1.default.info(`Created payment access token for payment ${paymentId}`, {
                    paymentId,
                    bookingId,
                    ipAddress,
                    tokenLength: token.length,
                    expiresAt
                });
                return token;
            }
            catch (error) {
                logger_1.default.error(`Failed to create payment access token: ${error.message}`, {
                    paymentId,
                    bookingId,
                    ipAddress
                });
                throw error;
            }
        });
    }
    /**
     * Verify payment access token and return payment/booking details
     */
    static verifyAccessToken(token, ipAddress) {
        return __awaiter(this, void 0, void 0, function* () {
            var _a;
            try {
                // Find the access token
                const accessToken = yield models_1.default.PaymentAccessToken.findOne({
                    where: { token },
                    include: [
                        {
                            model: models_1.default.Payment,
                            as: 'payment',
                            include: [
                                {
                                    model: models_1.default.Booking,
                                    as: 'booking',
                                    include: [
                                        { model: models_1.default.Guest, as: 'guest' },
                                        { model: models_1.default.Room, as: 'room' }
                                    ]
                                }
                            ]
                        }
                    ]
                });
                if (!accessToken) {
                    logger_1.default.warn(`Invalid payment access token attempted: ${token}`, { ipAddress });
                    return { isValid: false, error: 'Invalid access token' };
                }
                // Check if token is expired
                if (accessToken.isExpired()) {
                    logger_1.default.warn(`Expired payment access token attempted: ${token}`, { ipAddress });
                    return { isValid: false, error: 'Access token has expired' };
                }
                // Check if token has already been used (optional - can be removed for multiple access)
                if (accessToken.isUsed) {
                    logger_1.default.warn(`Already used payment access token attempted: ${token}`, { ipAddress });
                    return { isValid: false, error: 'Access token has already been used' };
                }
                // Verify IP address matches (with some flexibility for mobile networks)
                if (accessToken.ipAddress !== ipAddress) {
                    // Log warning but don't block - IP can change on mobile networks
                    logger_1.default.warn(`IP address mismatch for payment access token`, {
                        token,
                        originalIP: accessToken.ipAddress,
                        currentIP: ipAddress
                    });
                }
                // Mark token as used (optional)
                yield accessToken.update({
                    isUsed: true,
                    usedAt: new Date()
                });
                logger_1.default.info(`Payment access token verified successfully`, {
                    token,
                    paymentId: accessToken.paymentId,
                    bookingId: accessToken.bookingId,
                    ipAddress
                });
                return {
                    isValid: true,
                    booking: (_a = accessToken.payment) === null || _a === void 0 ? void 0 : _a.booking,
                    payment: accessToken.payment
                };
            }
            catch (error) {
                logger_1.default.error(`Error verifying payment access token: ${error.message}`, { token, ipAddress });
                return { isValid: false, error: 'Failed to verify access token' };
            }
        });
    }
    /**
     * Clean up expired access tokens
     */
    static cleanupExpiredTokens() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const deletedCount = yield models_1.default.PaymentAccessToken.destroy({
                    where: {
                        expiresAt: {
                            [sequelize_1.Op.lt]: new Date()
                        }
                    }
                });
                logger_1.default.info(`Cleaned up ${deletedCount} expired payment access tokens`);
                return deletedCount;
            }
            catch (error) {
                logger_1.default.error(`Error cleaning up expired payment access tokens: ${error.message}`);
                throw error;
            }
        });
    }
    /**
     * Get access token statistics
     */
    static getTokenStats() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const [total, active, expired, used] = yield Promise.all([
                    models_1.default.PaymentAccessToken.count(),
                    models_1.default.PaymentAccessToken.count({
                        where: {
                            isUsed: false,
                            expiresAt: {
                                [sequelize_1.Op.gt]: new Date()
                            }
                        }
                    }),
                    models_1.default.PaymentAccessToken.count({
                        where: {
                            expiresAt: {
                                [sequelize_1.Op.lt]: new Date()
                            }
                        }
                    }),
                    models_1.default.PaymentAccessToken.count({
                        where: { isUsed: true }
                    })
                ]);
                return { total, active, expired, used };
            }
            catch (error) {
                logger_1.default.error(`Error getting payment access token stats: ${error.message}`);
                throw error;
            }
        });
    }
}
exports.PaymentAccessTokenService = PaymentAccessTokenService;
exports.default = PaymentAccessTokenService;
