"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.refundPayment = exports.verifyPayment = exports.processWebhook = exports.getPaymentStatus = exports.initiatePayment = void 0;
const axios_1 = __importDefault(require("axios"));
const models_1 = __importDefault(require("../models"));
const logger_1 = __importDefault(require("../utils/logger"));
// Konnect API configuration
const KONNECT_API_BASE_URL = process.env.KONNECT_API_BASE_URL || 'https://api.konnect.network/api/v2';
const KONNECT_API_KEY = process.env.KONNECT_API_KEY || '';
const KONNECT_WALLET_ID = process.env.KONNECT_WALLET_ID || '';
// Validate required environment variables
if (!KONNECT_API_KEY || !KONNECT_WALLET_ID) {
    throw new Error('Konnect API key and wallet ID must be configured in environment variables');
}
// Konnect API client
const konnectApiClient = axios_1.default.create({
    baseURL: KONNECT_API_BASE_URL,
    headers: {
        'x-api-key': KONNECT_API_KEY,
        'Content-Type': 'application/json',
    },
});
/**
 * Initiate a payment with Konnect
 * @param amount - Payment amount
 * @param paymentMethod - Payment method (bank_card, wallet, e-DINAR)
 * @param customerDetails - Customer information
 * @param orderDetails - Order information
 * @returns Payment URL where customer can complete the transaction
 */
const initiatePayment = (amount, paymentMethod, customerDetails, orderDetails) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
    const paymentRequest = {
        receiverWalletId: KONNECT_WALLET_ID,
        token: 'TND', // Tunisian Dinar
        amount: amount * 1000, // Konnect expects amount in millimes (1 TND = 1000 millimes)
        type: 'immediate',
        description: orderDetails.description,
        acceptedPaymentMethods: [paymentMethod],
        lifespan: 10, // 10 minutes
        checkoutForm: true,
        addPaymentFeesToAmount: true,
        firstName: customerDetails.firstName,
        lastName: customerDetails.lastName,
        phoneNumber: customerDetails.phoneNumber,
        email: customerDetails.email,
        orderId: orderDetails.orderId,
        webhook: `${process.env.BASE_URL}/api/v1/payment-webhook`, // Webhook URL for payment status updates
        successUrl: `${process.env.FRONTEND_URL}/payment-success`,
        failUrl: `${process.env.FRONTEND_URL}/payment-failure`,
    };
    try {
        logger_1.default.info(`Initiating Konnect payment for order ${orderDetails.orderId}`, {
            amount,
            paymentMethod,
            customerEmail: customerDetails.email
        });
        const response = yield konnectApiClient.post('/payments/init-payment', paymentRequest);
        // Log the actual response structure for debugging
        logger_1.default.info(`Konnect API response for order ${orderDetails.orderId}`, {
            status: response.status,
            statusText: response.statusText,
            responseData: response.data
        });
        // Safely access response properties
        const paymentId = ((_a = response.data) === null || _a === void 0 ? void 0 : _a.paymentRef) || 'unknown';
        const paymentLink = ((_b = response.data) === null || _b === void 0 ? void 0 : _b.payUrl) || 'unknown';
        logger_1.default.info(`Successfully initiated Konnect payment for order ${orderDetails.orderId}`, {
            paymentId,
            paymentLink
        });
        return response.data;
    }
    catch (error) {
        // Enhanced error logging for Konnect API responses
        const errorDetails = {
            message: error.message,
            status: (_c = error.response) === null || _c === void 0 ? void 0 : _c.status,
            statusText: (_d = error.response) === null || _d === void 0 ? void 0 : _d.statusText,
            data: (_e = error.response) === null || _e === void 0 ? void 0 : _e.data,
            headers: (_f = error.response) === null || _f === void 0 ? void 0 : _f.headers,
            config: {
                url: (_g = error.config) === null || _g === void 0 ? void 0 : _g.url,
                method: (_h = error.config) === null || _h === void 0 ? void 0 : _h.method,
                headers: (_j = error.config) === null || _j === void 0 ? void 0 : _j.headers
            }
        };
        logger_1.default.error(`Failed to initiate Konnect payment for order ${orderDetails.orderId}`, errorDetails);
        // Include API response details in thrown error
        const apiErrorMessage = ((_l = (_k = error.response) === null || _k === void 0 ? void 0 : _k.data) === null || _l === void 0 ? void 0 : _l.message) || ((_o = (_m = error.response) === null || _m === void 0 ? void 0 : _m.data) === null || _o === void 0 ? void 0 : _o.error) || error.message;
        throw new Error(`Failed to initiate Konnect payment: ${apiErrorMessage}`);
    }
});
exports.initiatePayment = initiatePayment;
/**
 * Get payment status from Konnect
 * @param paymentId - Konnect payment ID
 * @returns Payment status information
 */
const getPaymentStatus = (paymentId) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const response = yield konnectApiClient.get(`/payments/${paymentId}`);
        return response.data;
    }
    catch (error) {
        throw new Error(`Failed to get Konnect payment status: ${error.message || error}`);
    }
});
exports.getPaymentStatus = getPaymentStatus;
/**
 * Process webhook notification from Konnect
 * @param paymentRef - Payment reference from webhook
 * @returns Updated payment record
 */
const processWebhook = (paymentRef) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f, _g;
    try {
        logger_1.default.info(`Processing Konnect webhook for payment reference: ${paymentRef}`);
        // Get payment status from Konnect
        const paymentStatus = yield (0, exports.getPaymentStatus)(paymentRef);
        logger_1.default.info(`Konnect payment status for ${paymentRef}: ${paymentStatus.payment.status}`);
        // Find our payment record by transactionId
        const payment = yield models_1.default.Payment.findOne({
            where: { transactionId: paymentRef }
        });
        if (!payment) {
            logger_1.default.warn(`Payment with reference ${paymentRef} not found in database`);
            return null;
        }
        // Check if payment status has actually changed to prevent unnecessary updates
        const konnectStatus = paymentStatus.payment.status;
        if (payment.status === konnectStatus) {
            logger_1.default.info(`Payment ${paymentRef} status unchanged (${konnectStatus}), skipping update`);
            return payment;
        }
        // Update payment status based on Konnect response
        let newStatus = 'pending';
        switch (konnectStatus) {
            case 'completed':
                newStatus = 'completed';
                break;
            case 'failed':
            case 'expired':
                newStatus = 'failed';
                break;
            case 'refunded':
                newStatus = 'refunded';
                break;
            default:
                newStatus = 'pending';
        }
        // Update payment in database
        yield payment.update({ status: newStatus });
        logger_1.default.info(`Successfully updated payment ${paymentRef} status to ${newStatus}`);
        return payment;
    }
    catch (error) {
        // Enhanced error logging for webhook processing
        const errorDetails = {
            message: error.message,
            status: (_a = error.response) === null || _a === void 0 ? void 0 : _a.status,
            statusText: (_b = error.response) === null || _b === void 0 ? void 0 : _b.statusText,
            data: (_c = error.response) === null || _c === void 0 ? void 0 : _c.data,
            stack: error.stack
        };
        logger_1.default.error(`Failed to process Konnect webhook for ${paymentRef}`, errorDetails);
        // Include API response details in thrown error if available
        const apiErrorMessage = ((_e = (_d = error.response) === null || _d === void 0 ? void 0 : _d.data) === null || _e === void 0 ? void 0 : _e.message) || ((_g = (_f = error.response) === null || _f === void 0 ? void 0 : _f.data) === null || _g === void 0 ? void 0 : _g.error) || error.message;
        throw new Error(`Failed to process Konnect webhook: ${apiErrorMessage}`);
    }
});
exports.processWebhook = processWebhook;
/**
 * Verify a payment's status with Konnect and update the local database.
 * This is used for on-demand checks, for instance when a user is waiting on a pending page.
 * @param paymentRef - The payment reference (transactionId) to verify.
 * @returns The updated payment record from the database.
 */
const verifyPayment = (paymentRef) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        logger_1.default.info(`Verifying payment status for reference: ${paymentRef}`);
        const paymentStatus = yield (0, exports.getPaymentStatus)(paymentRef);
        const konnectStatus = paymentStatus.payment.status;
        const payment = yield models_1.default.Payment.findOne({ where: { transactionId: paymentRef } });
        if (!payment) {
            logger_1.default.warn(`Payment with reference ${paymentRef} not found during verification.`);
            // We don't throw an error, as the payment might not have been created yet.
            return null;
        }
        // Map Konnect status to our internal status
        let newStatus = 'pending';
        switch (konnectStatus) {
            case 'completed':
                newStatus = 'completed';
                break;
            case 'failed':
            case 'expired':
                newStatus = 'failed';
                break;
            case 'refunded':
                newStatus = 'refunded';
                break;
            default:
                newStatus = 'pending';
        }
        // Only update if the status has changed
        if (payment.status !== newStatus) {
            yield payment.update({ status: newStatus });
            logger_1.default.info(`Payment ${paymentRef} status verified and updated to ${newStatus}`);
        }
        else {
            logger_1.default.info(`Payment ${paymentRef} status is already up-to-date (${newStatus})`);
        }
        return payment;
    }
    catch (error) {
        const errorDetails = {
            message: error.message,
            status: (_a = error.response) === null || _a === void 0 ? void 0 : _a.status,
            data: (_b = error.response) === null || _b === void 0 ? void 0 : _b.data,
        };
        logger_1.default.error(`Failed to verify payment ${paymentRef}`, errorDetails);
        throw new Error(`Failed to verify payment status: ${error.message}`);
    }
});
exports.verifyPayment = verifyPayment;
/**
 * Refund a payment through Konnect
 * @param transactionId - Konnect payment ID
 * @param amount - Amount to refund
 * @returns Refund status
 */
const refundPayment = (transactionId, amount) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
    try {
        logger_1.default.info(`Initiating Konnect refund for payment ${transactionId}`, { amount });
        const response = yield konnectApiClient.post('/refunds', {
            paymentId: transactionId,
            amount: amount * 1000, // Konnect expects amount in millimes
        });
        logger_1.default.info(`Konnect refund response for payment ${transactionId}`, {
            refundId: (_a = response.data.refund) === null || _a === void 0 ? void 0 : _a.id,
            refundStatus: (_b = response.data.refund) === null || _b === void 0 ? void 0 : _b.status
        });
        // Return a properly formatted response
        const refundResponse = {
            refund: {
                id: ((_c = response.data.refund) === null || _c === void 0 ? void 0 : _c.id) || 'refund_id',
                status: ((_d = response.data.refund) === null || _d === void 0 ? void 0 : _d.status) || 'completed',
                paymentId: transactionId,
                amount: amount
            }
        };
        return refundResponse;
    }
    catch (error) {
        // Enhanced error logging for Konnect API responses
        const errorDetails = {
            message: error.message,
            status: (_e = error.response) === null || _e === void 0 ? void 0 : _e.status,
            statusText: (_f = error.response) === null || _f === void 0 ? void 0 : _f.statusText,
            data: (_g = error.response) === null || _g === void 0 ? void 0 : _g.data,
            headers: (_h = error.response) === null || _h === void 0 ? void 0 : _h.headers,
            config: {
                url: (_j = error.config) === null || _j === void 0 ? void 0 : _j.url,
                method: (_k = error.config) === null || _k === void 0 ? void 0 : _k.method,
                headers: (_l = error.config) === null || _l === void 0 ? void 0 : _l.headers
            }
        };
        logger_1.default.error(`Failed to refund Konnect payment ${transactionId}`, errorDetails);
        // Include API response details in thrown error
        const apiErrorMessage = ((_o = (_m = error.response) === null || _m === void 0 ? void 0 : _m.data) === null || _o === void 0 ? void 0 : _o.message) || ((_q = (_p = error.response) === null || _p === void 0 ? void 0 : _p.data) === null || _q === void 0 ? void 0 : _q.error) || error.message;
        throw new Error(`Failed to refund Konnect payment: ${apiErrorMessage}`);
    }
});
exports.refundPayment = refundPayment;
