"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CleanupService = void 0;
const sequelize_1 = require("sequelize");
const konnectPayment_service_1 = require("./konnectPayment.service");
const models_1 = __importDefault(require("../models"));
const paymentAccessToken_service_1 = __importDefault(require("./paymentAccessToken.service"));
const logger_1 = __importDefault(require("../utils/logger"));
/**
 * Cleanup Service
 * Handles daily cleanup tasks like resetting rate limits
 */
class CleanupService {
    /**
     * Delete all payment attempts to reset daily rate limits
     */
    static cleanupPaymentAttempts() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                logger_1.default.info('Starting cleanup of payment attempts');
                const deletedCount = yield models_1.default.PaymentAttempt.destroy({
                    where: {},
                    truncate: true
                });
                logger_1.default.info(`Cleanup completed: Deleted ${deletedCount} payment attempts`);
            }
            catch (error) {
                logger_1.default.error(`Error during payment attempts cleanup: ${error.message}`, { error });
                throw error;
            }
        });
    }
    /**
     * Delete all booking attempts to reset daily rate limits
     */
    static cleanupBookingAttempts() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                logger_1.default.info('Starting cleanup of booking attempts');
                const deletedCount = yield models_1.default.BookingAttempt.destroy({
                    where: {},
                    truncate: true
                });
                logger_1.default.info(`Cleanup completed: Deleted ${deletedCount} booking attempts`);
            }
            catch (error) {
                logger_1.default.error(`Error during booking attempts cleanup: ${error.message}`, { error });
                throw error;
            }
        });
    }
    /**
     * Clean up expired payment access tokens
     */
    static cleanupPaymentAccessTokens() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                logger_1.default.info('Starting cleanup of expired payment access tokens');
                const deletedCount = yield paymentAccessToken_service_1.default.cleanupExpiredTokens();
                logger_1.default.info(`Cleanup completed: Deleted ${deletedCount} expired payment access tokens`);
            }
            catch (error) {
                logger_1.default.error(`Error during payment access tokens cleanup: ${error.message}`, { error });
                throw error;
            }
        });
    }
    /**
     * Deletes pending bookings that are older than 15 minutes.
     * This is to clean up abandoned checkouts.
     */
    static cleanupPendingBookings() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const elevenMinutesAgo = new Date(Date.now() - 11 * 60 * 1000);
                const pendingPayments = yield models_1.default.Payment.findAll({
                    where: {
                        status: 'pending',
                        createdAt: { [sequelize_1.Op.lt]: elevenMinutesAgo },
                        transaction_id: { [sequelize_1.Op.ne]: null }, // Ensure we only check payments sent to Konnect
                    },
                    include: [{ model: models_1.default.Booking, as: 'booking' }],
                });
                if (pendingPayments.length === 0) {
                    return; // Nothing to do
                }
                logger_1.default.info(`Found ${pendingPayments.length} potentially stale pending payments to verify.`);
                for (const payment of pendingPayments) {
                    try {
                        // The transactionId from our DB is the paymentRef for Konnect
                        const konnectStatus = yield (0, konnectPayment_service_1.getPaymentStatus)(payment.transactionId);
                        const booking = payment.booking;
                        if (!booking) {
                            logger_1.default.warn(`Payment ${payment.id} has no associated booking, skipping.`);
                            continue;
                        }
                        switch (konnectStatus.payment.status) {
                            case 'completed':
                                logger_1.default.info(`Konnect confirmed payment ${payment.transactionId} is 'completed'. Updating booking ${booking.id}.`);
                                yield payment.update({ status: 'completed' });
                                yield booking.update({ status: 'confirmed' });
                                break;
                            case 'failed':
                            case 'expired':
                                logger_1.default.warn(`Konnect confirmed payment ${payment.transactionId} is '${konnectStatus.payment.status}'. Deleting booking ${booking.id}.`);
                                yield booking.destroy(); // This will also cascade-delete the payment record
                                break;
                            case 'pending':
                                logger_1.default.info(`Konnect reports payment ${payment.transactionId} is still 'pending'. No action taken.`);
                                break;
                            default:
                                logger_1.default.warn(`Received unknown status '${konnectStatus.payment.status}' for payment ${payment.transactionId}.`);
                        }
                    }
                    catch (error) {
                        logger_1.default.error(`Error verifying payment ${payment.transactionId} with Konnect: ${error.message}`, { error });
                    }
                }
            }
            catch (error) {
                logger_1.default.error(`Error during pending payments cleanup: ${error.message}`, { error });
                // Do not rethrow, as this is a background task and shouldn't crash the scheduler
            }
        });
    }
    /**
     * Run all daily cleanup tasks
     */
    static runDailyCleanup() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                logger_1.default.info('Starting daily cleanup tasks');
                yield Promise.all([
                    CleanupService.cleanupPaymentAttempts(),
                    CleanupService.cleanupBookingAttempts(),
                    CleanupService.cleanupPaymentAccessTokens()
                ]);
                logger_1.default.info('Daily cleanup tasks completed successfully');
            }
            catch (error) {
                logger_1.default.error(`Error during daily cleanup: ${error.message}`, { error });
                throw error;
            }
        });
    }
    /**
     * Get cleanup statistics
     */
    static getCleanupStats() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const [paymentAttempts, bookingAttempts] = yield Promise.all([
                    models_1.default.PaymentAttempt.count(),
                    models_1.default.BookingAttempt.count()
                ]);
                return {
                    paymentAttempts,
                    bookingAttempts
                };
            }
            catch (error) {
                logger_1.default.error(`Error getting cleanup stats: ${error.message}`, { error });
                throw error;
            }
        });
    }
}
exports.CleanupService = CleanupService;
exports.default = CleanupService;
