"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const sequelize_1 = require("sequelize");
const models_1 = __importDefault(require("../models"));
const email_service_1 = __importDefault(require("./email.service"));
const feedback_controller_1 = require("../controllers/feedback.controller");
/**
 * Service to handle checkout-related automation
 */
class CheckoutService {
    /**
     * Process all bookings that have checkout date = today
     * Send feedback emails to guests who checked out today
     */
    /**
     * Find confirmed bookings with a checkout date in the past and mark them as 'completed'.
     * This should run before processing feedback to ensure statuses are up-to-date.
     */
    completePastCheckouts() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const today = new Date();
                const endOfToday = new Date(today.getFullYear(), today.getMonth(), today.getDate(), 23, 59, 59);
                console.log(`Checking for bookings to mark as completed for checkout dates <= ${endOfToday.toDateString()}`);
                const [updateCount] = yield models_1.default.Booking.update({ status: 'completed' }, {
                    where: {
                        checkOutDate: {
                            [sequelize_1.Op.lte]: endOfToday,
                        },
                        status: 'confirmed',
                    },
                });
                if (updateCount > 0) {
                    console.log(`${updateCount} bookings were marked as completed.`);
                }
                else {
                    console.log('No bookings needed to be marked as completed.');
                }
            }
            catch (error) {
                console.error('Error completing past checkout bookings:', error);
                throw error;
            }
        });
    }
    processCheckoutFeedback() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const today = new Date();
                // Set to start of day (00:00:00)
                const startOfDay = new Date(today.getFullYear(), today.getMonth(), today.getDate());
                // Set to end of day (23:59:59)
                const endOfDay = new Date(today.getFullYear(), today.getMonth(), today.getDate(), 23, 59, 59);
                console.log(`Processing checkout feedback for date: ${startOfDay.toDateString()}`);
                // Find all bookings that checkout today and are confirmed/completed
                const checkoutBookings = yield models_1.default.Booking.findAll({
                    where: {
                        checkOutDate: {
                            [sequelize_1.Op.between]: [startOfDay, endOfDay]
                        },
                        status: {
                            [sequelize_1.Op.in]: ['confirmed', 'completed']
                        },
                        // Only send feedback if not already sent (feedbackSent flag)
                        feedbackSent: {
                            [sequelize_1.Op.or]: [false, null]
                        }
                    },
                    include: [
                        {
                            model: models_1.default.Guest,
                            as: 'guest',
                            attributes: ['id', 'firstName', 'lastName', 'email']
                        },
                        {
                            model: models_1.default.Room,
                            as: 'room',
                            attributes: ['id', 'number'],
                            include: [
                                {
                                    model: models_1.default.RoomType,
                                    as: 'roomType',
                                    attributes: ['name']
                                }
                            ]
                        }
                    ]
                });
                console.log(`Found ${checkoutBookings.length} bookings for checkout feedback`);
                let successCount = 0;
                let errorCount = 0;
                // Process each booking
                for (const booking of checkoutBookings) {
                    try {
                        yield this.sendCheckoutFeedbackEmail(booking);
                        // Mark feedback as sent
                        yield booking.update({ feedbackSent: true });
                        successCount++;
                        console.log(`✓ Feedback email sent for booking ${booking.id} to ${booking.guest.email}`);
                    }
                    catch (error) {
                        errorCount++;
                        console.error(`✗ Failed to send feedback email for booking ${booking.id}:`, error);
                        // Continue processing other bookings even if one fails
                    }
                }
                console.log(`Checkout feedback processing completed: ${successCount} sent, ${errorCount} failed`);
            }
            catch (error) {
                console.error('Error processing checkout feedback:', error);
                throw error;
            }
        });
    }
    /**
     * Send feedback email for a specific booking
     */
    sendCheckoutFeedbackEmail(booking) {
        return __awaiter(this, void 0, void 0, function* () {
            const guest = booking.guest;
            const room = booking.room;
            if (!guest || !guest.email) {
                throw new Error(`No guest email found for booking ${booking.id}`);
            }
            // Generate unique feedback token
            const feedbackToken = (0, feedback_controller_1.generateFeedbackToken)();
            // Store feedback token in database for validation
            yield models_1.default.FeedbackToken.create({
                token: feedbackToken,
                bookingId: booking.id,
                guestId: guest.id,
                expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // 30 days from now
                isUsed: false
            });
            // Prepare room name
            const roomName = room.roomType ? room.roomType.name : `Room ${room.number}`;
            // Send feedback request email
            yield email_service_1.default.sendFeedbackRequest(guest.email, `${guest.firstName} ${guest.lastName}`, roomName, booking.checkInDate.toString(), booking.checkOutDate.toString(), feedbackToken);
        });
    }
    /**
     * Get statistics about checkout feedback processing
     */
    getCheckoutFeedbackStats() {
        return __awaiter(this, arguments, void 0, function* (days = 7) {
            try {
                const startDate = new Date();
                startDate.setDate(startDate.getDate() - days);
                const totalCheckouts = yield models_1.default.Booking.count({
                    where: {
                        checkOutDate: {
                            [sequelize_1.Op.gte]: startDate
                        },
                        status: {
                            [sequelize_1.Op.in]: ['confirmed', 'completed']
                        }
                    }
                });
                const feedbackSent = yield models_1.default.Booking.count({
                    where: {
                        checkOutDate: {
                            [sequelize_1.Op.gte]: startDate
                        },
                        status: {
                            [sequelize_1.Op.in]: ['confirmed', 'completed']
                        },
                        feedbackSent: true
                    }
                });
                const feedbackPending = totalCheckouts - feedbackSent;
                const feedbackRate = totalCheckouts > 0 ? (feedbackSent / totalCheckouts) * 100 : 0;
                return {
                    totalCheckouts,
                    feedbackSent,
                    feedbackPending,
                    feedbackRate: Math.round(feedbackRate * 100) / 100
                };
            }
            catch (error) {
                console.error('Error getting checkout feedback stats:', error);
                throw error;
            }
        });
    }
}
exports.default = new CheckoutService();
