"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sequelize_1 = require("sequelize");
exports.default = (sequelize) => {
    const Room = sequelize.define('Room', {
        id: {
            type: sequelize_1.DataTypes.INTEGER,
            autoIncrement: true,
            primaryKey: true,
        },
        roomTypeId: {
            type: sequelize_1.DataTypes.INTEGER,
            allowNull: false,
            field: 'room_type_id',
            references: {
                model: 'room_types',
                key: 'id',
            },
        },
        number: {
            type: sequelize_1.DataTypes.STRING(10),
            allowNull: false,
            unique: true,
        },
        floor: {
            type: sequelize_1.DataTypes.STRING(10),
            allowNull: false,
        },
        capacity: {
            type: sequelize_1.DataTypes.INTEGER,
            allowNull: false,
        },
        pricePerNight: {
            type: sequelize_1.DataTypes.DECIMAL(10, 2),
            allowNull: false,
            field: 'price_per_night',
        },
        discount: {
            type: sequelize_1.DataTypes.DECIMAL(5, 2),
            allowNull: true,
            defaultValue: 0,
            validate: {
                min: 0,
                max: 100,
            }
        },
        status: {
            type: sequelize_1.DataTypes.ENUM('available', 'occupied', 'maintenance', 'reserved'),
            allowNull: false,
            defaultValue: 'available',
        },
        description: {
            type: sequelize_1.DataTypes.TEXT,
            allowNull: true,
        },
        images: {
            type: sequelize_1.DataTypes.JSON,
            allowNull: true,
            defaultValue: [],
            get() {
                // Get the raw value
                const rawValue = this.getDataValue('images');
                // If it's not set or not an array, return the raw value
                if (!rawValue || !Array.isArray(rawValue)) {
                    return rawValue;
                }
                // Transform each filename to a full path
                return rawValue.map((filename) => `/uploads/rooms/${filename}`);
            }
        },
        createdAt: {
            type: sequelize_1.DataTypes.DATE,
            field: 'created_at',
        },
        updatedAt: {
            type: sequelize_1.DataTypes.DATE,
            field: 'updated_at',
        },
    }, {
        tableName: 'rooms',
        timestamps: true,
    });
    // Define associations
    Room.associate = (models) => {
        Room.belongsTo(models.RoomType, { foreignKey: 'roomTypeId', as: 'roomType' });
        Room.hasMany(models.Booking, { foreignKey: 'roomId', as: 'bookings' });
        Room.belongsToMany(models.Amenity, {
            through: 'RoomAmenity',
            foreignKey: 'roomId',
            otherKey: 'amenityId',
            as: 'amenities'
        });
    };
    return Room;
};
