"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sequelize_1 = require("sequelize");
// PaymentAccessToken model class
class PaymentAccessToken extends sequelize_1.Model {
    // Instance methods
    isExpired() {
        return new Date() > this.expiresAt;
    }
    isValid() {
        return !this.isUsed && !this.isExpired();
    }
    markAsUsed() {
        this.isUsed = true;
        this.usedAt = new Date();
    }
}
// Model factory function
exports.default = (sequelize) => {
    PaymentAccessToken.init({
        id: {
            type: sequelize_1.DataTypes.INTEGER,
            autoIncrement: true,
            primaryKey: true,
        },
        token: {
            type: sequelize_1.DataTypes.STRING(64),
            allowNull: false,
            unique: true,
            comment: 'Unique access token for payment result pages',
        },
        paymentId: {
            type: sequelize_1.DataTypes.INTEGER,
            allowNull: false,
            references: {
                model: 'payments',
                key: 'id',
            },
            onUpdate: 'CASCADE',
            onDelete: 'CASCADE',
            comment: 'Reference to the payment',
        },
        bookingId: {
            type: sequelize_1.DataTypes.INTEGER,
            allowNull: false,
            references: {
                model: 'bookings',
                key: 'id',
            },
            onUpdate: 'CASCADE',
            onDelete: 'CASCADE',
            comment: 'Reference to the booking',
        },
        ipAddress: {
            type: sequelize_1.DataTypes.STRING(45), // Support both IPv4 and IPv6
            allowNull: false,
            comment: 'IP address of the user who initiated the payment',
        },
        userAgent: {
            type: sequelize_1.DataTypes.TEXT,
            allowNull: false,
            defaultValue: '',
            comment: 'User agent of the client who initiated the payment',
        },
        expiresAt: {
            type: sequelize_1.DataTypes.DATE,
            allowNull: false,
            comment: 'Token expiration timestamp (24 hours from creation)',
        },
        isUsed: {
            type: sequelize_1.DataTypes.BOOLEAN,
            allowNull: false,
            defaultValue: false,
            comment: 'Whether the token has been used to access payment results',
        },
        usedAt: {
            type: sequelize_1.DataTypes.DATE,
            allowNull: true,
            comment: 'Timestamp when the token was first used',
        },
        createdAt: {
            type: sequelize_1.DataTypes.DATE,
            allowNull: false,
            defaultValue: sequelize_1.DataTypes.NOW,
        },
        updatedAt: {
            type: sequelize_1.DataTypes.DATE,
            allowNull: false,
            defaultValue: sequelize_1.DataTypes.NOW,
        },
    }, {
        sequelize,
        tableName: 'payment_access_tokens',
        timestamps: true,
        indexes: [
            {
                fields: ['token'],
                unique: true,
            },
            {
                fields: ['paymentId'],
            },
            {
                fields: ['bookingId'],
            },
            {
                fields: ['ipAddress'],
            },
            {
                fields: ['expiresAt'],
            },
            {
                fields: ['isUsed'],
            },
        ],
    });
    return PaymentAccessToken;
};
