"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sequelize_1 = require("sequelize");
exports.default = (sequelize) => {
    const Booking = sequelize.define('Booking', {
        id: {
            type: sequelize_1.DataTypes.INTEGER,
            autoIncrement: true,
            primaryKey: true,
        },
        roomId: {
            type: sequelize_1.DataTypes.INTEGER,
            allowNull: false,
            field: 'room_id',
            references: {
                model: 'rooms',
                key: 'id',
            },
        },
        guestId: {
            type: sequelize_1.DataTypes.INTEGER,
            allowNull: false,
            field: 'guest_id',
            references: {
                model: 'guests',
                key: 'id',
            },
        },
        checkInDate: {
            type: sequelize_1.DataTypes.DATEONLY,
            allowNull: false,
            field: 'check_in_date',
        },
        checkOutDate: {
            type: sequelize_1.DataTypes.DATEONLY,
            allowNull: false,
            field: 'check_out_date',
        },
        totalPrice: {
            type: sequelize_1.DataTypes.DECIMAL(10, 2),
            allowNull: true, // or false if it should always be calculated on creation
            field: 'total_price',
        },
        status: {
            type: sequelize_1.DataTypes.ENUM('confirmed', 'pending', 'cancelled', 'completed'),
            allowNull: false,
            defaultValue: 'pending',
        },
        specialRequests: {
            type: sequelize_1.DataTypes.TEXT,
            allowNull: true,
            field: 'special_requests',
        },
        feedbackSent: {
            type: sequelize_1.DataTypes.BOOLEAN,
            allowNull: true,
            defaultValue: false,
            field: 'feedback_sent',
        },
        externalId: {
            type: sequelize_1.DataTypes.STRING,
            allowNull: true,
            field: 'external_id',
        },
        paymentStatus: {
            type: sequelize_1.DataTypes.ENUM('pending', 'paid', 'failed', 'refunded'),
            allowNull: true,
            field: 'payment_status',
        },
        source: {
            type: sequelize_1.DataTypes.STRING,
            allowNull: true,
            field: 'source',
        },
        notes: {
            type: sequelize_1.DataTypes.TEXT,
            allowNull: true,
            field: 'notes',
        },
        createdBy: {
            type: sequelize_1.DataTypes.INTEGER,
            allowNull: true,
            field: 'created_by',
            references: {
                model: 'users',
                key: 'id',
            },
        },
        createdAt: {
            type: sequelize_1.DataTypes.DATE,
            field: 'created_at',
        },
        updatedAt: {
            type: sequelize_1.DataTypes.DATE,
            field: 'updated_at',
        },
    }, {
        tableName: 'bookings',
        timestamps: true,
        validate: {
            checkInBeforeCheckOut() {
                if (this.checkInDate >= this.checkOutDate) {
                    throw new Error('Check-in date must be before check-out date');
                }
            },
        },
    });
    // Define associations
    Booking.associate = (models) => {
        Booking.belongsTo(models.Room, { foreignKey: 'roomId', as: 'room' });
        Booking.belongsTo(models.Guest, { foreignKey: 'guestId', as: 'guest' });
        Booking.belongsTo(models.User, { foreignKey: 'createdBy', as: 'creator' });
        Booking.hasMany(models.Payment, {
            foreignKey: 'bookingId',
            as: 'payments',
            onDelete: 'CASCADE',
            hooks: true
        });
    };
    return Booking;
};
