"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.cleanupOldAttempts = exports.paymentRateLimit = exports.recordPaymentAttempt = void 0;
const sequelize_1 = require("sequelize");
const models_1 = __importDefault(require("../models"));
const errorHandler_1 = require("../utils/errorHandler");
const logger_1 = __importDefault(require("../utils/logger"));
// Normalize IP address for consistent storage and comparison
const normalizeIP = (ip) => {
    if (!ip)
        return '127.0.0.1';
    // Clean up IPv6 mapped IPv4 addresses
    if (ip.startsWith('::ffff:')) {
        ip = ip.substring(7);
    }
    // Convert IPv6 localhost to IPv4 localhost for consistency
    if (ip === '::1') {
        return '127.0.0.1';
    }
    return ip;
};
// Get client IP address from request
const getClientIP = (req) => {
    var _a, _b, _c, _d;
    // Try multiple sources for the real IP address
    const forwarded = req.headers['x-forwarded-for'];
    const realIP = req.headers['x-real-ip'];
    const cfConnectingIP = req.headers['cf-connecting-ip']; // Cloudflare
    let ip = forwarded ? forwarded.split(',')[0].trim() :
        realIP ||
            cfConnectingIP ||
            req.ip ||
            ((_a = req.connection) === null || _a === void 0 ? void 0 : _a.remoteAddress) ||
            ((_b = req.socket) === null || _b === void 0 ? void 0 : _b.remoteAddress) ||
            ((_d = (_c = req.connection) === null || _c === void 0 ? void 0 : _c.socket) === null || _d === void 0 ? void 0 : _d.remoteAddress);
    return normalizeIP(ip || '127.0.0.1');
};
// Check failed payment attempts (10 failed attempts limit)
const checkFailedAttempts = (ipAddress) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Get the most recent successful payment for this IP
        const lastSuccessfulPayment = yield models_1.default.PaymentAttempt.findOne({
            where: {
                ipAddress,
                isSuccessful: true,
            },
            order: [['attemptedAt', 'DESC']],
        });
        // Count failed attempts since last successful payment (or all time if no successful payment)
        const whereCondition = {
            ipAddress,
            isSuccessful: false,
        };
        if (lastSuccessfulPayment) {
            whereCondition.attemptedAt = {
                [sequelize_1.Op.gt]: lastSuccessfulPayment.attemptedAt,
            };
        }
        const failedAttempts = yield models_1.default.PaymentAttempt.count({
            where: whereCondition,
        });
        logger_1.default.info(`IP ${ipAddress} has ${failedAttempts} failed payment attempts since last success`);
        return failedAttempts >= 10;
    }
    catch (error) {
        logger_1.default.error(`Error checking failed attempts for IP ${ipAddress}: ${error.message}`);
        return false;
    }
});
// Check daily payment attempts (20 total attempts per day limit)
const checkDailyAttempts = (ipAddress) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        const tomorrow = new Date(today);
        tomorrow.setDate(tomorrow.getDate() + 1);
        const dailyAttempts = yield models_1.default.PaymentAttempt.count({
            where: {
                ipAddress,
                attemptedAt: {
                    [sequelize_1.Op.gte]: today,
                    [sequelize_1.Op.lt]: tomorrow,
                },
            },
        });
        logger_1.default.info(`IP ${ipAddress} has ${dailyAttempts} total payment attempts today`);
        return dailyAttempts >= 20;
    }
    catch (error) {
        logger_1.default.error(`Error checking daily attempts for IP ${ipAddress}: ${error.message}`);
        return false;
    }
});
// Record payment attempt
const recordPaymentAttempt = (ipAddress, isSuccessful, paymentId, userAgent) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        yield models_1.default.PaymentAttempt.create({
            ipAddress,
            isSuccessful,
            attemptedAt: new Date(),
            paymentId,
            userAgent,
        });
        logger_1.default.info(`Recorded payment attempt for IP ${ipAddress}`, {
            isSuccessful,
            paymentId,
            ipAddress,
        });
    }
    catch (error) {
        logger_1.default.error(`Error recording payment attempt for IP ${ipAddress}: ${error.message}`);
    }
});
exports.recordPaymentAttempt = recordPaymentAttempt;
// Main rate limiting middleware
const paymentRateLimit = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const ipAddress = getClientIP(req);
        logger_1.default.info(`Checking payment rate limits for IP: ${ipAddress}`);
        // Check failed attempts limit (10 failed attempts)
        const hasExceededFailedLimit = yield checkFailedAttempts(ipAddress);
        if (hasExceededFailedLimit) {
            logger_1.default.warn(`IP ${ipAddress} blocked due to too many failed payment attempts`);
            return next(new errorHandler_1.AppError('Too many failed payment attempts. Please try again later or contact support.', 429));
        }
        // Check daily attempts limit (20 total attempts per day)
        const hasExceededDailyLimit = yield checkDailyAttempts(ipAddress);
        if (hasExceededDailyLimit) {
            logger_1.default.warn(`IP ${ipAddress} blocked due to daily rate limit exceeded`);
            return next(new errorHandler_1.AppError('Rate limit exceeded. Maximum 20 payment attempts per day allowed.', 429));
        }
        // Store IP in request for later use
        req.clientIP = ipAddress;
        next();
    }
    catch (error) {
        logger_1.default.error(`Error in payment rate limit middleware: ${error.message}`);
        // Don't block the request if there's an error in rate limiting
        next();
    }
});
exports.paymentRateLimit = paymentRateLimit;
// Cleanup old payment attempts (optional - can be called periodically)
const cleanupOldAttempts = (...args_1) => __awaiter(void 0, [...args_1], void 0, function* (daysToKeep = 30) {
    try {
        const cutoffDate = new Date();
        cutoffDate.setDate(cutoffDate.getDate() - daysToKeep);
        const deletedCount = yield models_1.default.PaymentAttempt.destroy({
            where: {
                attemptedAt: {
                    [sequelize_1.Op.lt]: cutoffDate,
                },
            },
        });
        logger_1.default.info(`Cleaned up ${deletedCount} old payment attempts older than ${daysToKeep} days`);
    }
    catch (error) {
        logger_1.default.error(`Error cleaning up old payment attempts: ${error.message}`);
    }
});
exports.cleanupOldAttempts = cleanupOldAttempts;
