"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.bookingRateLimit = exports.recordBookingAttempt = void 0;
const models_1 = __importDefault(require("../models"));
const logger_1 = __importDefault(require("../utils/logger"));
const sequelize_1 = require("sequelize");
/**
 * Record a booking attempt in the database
 */
const recordBookingAttempt = (ipAddress_1, isSuccessful_1, bookingId_1, ...args_1) => __awaiter(void 0, [ipAddress_1, isSuccessful_1, bookingId_1, ...args_1], void 0, function* (ipAddress, isSuccessful, bookingId, userAgent = '') {
    try {
        yield models_1.default.BookingAttempt.create({
            ipAddress,
            isSuccessful,
            attemptedAt: new Date(),
            bookingId,
            userAgent,
        });
        logger_1.default.info(`Recorded booking attempt for IP ${ipAddress}`, {
            ipAddress,
            isSuccessful,
            bookingId,
        });
    }
    catch (error) {
        logger_1.default.error(`Failed to record booking attempt: ${error.message}`);
    }
});
exports.recordBookingAttempt = recordBookingAttempt;
/**
 * Rate limiting middleware for booking creation
 * Limits: 5 successful bookings per day per IP
 */
const bookingRateLimit = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    try {
        // Get client IP address
        const clientIP = req.clientIP ||
            req.ip ||
            req.connection.remoteAddress ||
            req.socket.remoteAddress ||
            ((_b = (_a = req.connection) === null || _a === void 0 ? void 0 : _a.socket) === null || _b === void 0 ? void 0 : _b.remoteAddress) ||
            '127.0.0.1';
        logger_1.default.info(`Checking booking rate limits for IP: ${clientIP}`);
        // Get start and end of today in local timezone
        const today = new Date();
        const startOfDay = new Date(today.getFullYear(), today.getMonth(), today.getDate());
        const endOfDay = new Date(today.getFullYear(), today.getMonth(), today.getDate() + 1);
        // Count successful bookings today
        const successfulBookingsToday = yield models_1.default.BookingAttempt.count({
            where: {
                ipAddress: clientIP,
                isSuccessful: true,
                attemptedAt: {
                    [sequelize_1.Op.gte]: startOfDay,
                    [sequelize_1.Op.lt]: endOfDay,
                },
            },
        });
        logger_1.default.info(`IP ${clientIP} has ${successfulBookingsToday} successful bookings today`);
        // Check if limit exceeded
        const DAILY_BOOKING_LIMIT = 5;
        if (successfulBookingsToday >= DAILY_BOOKING_LIMIT) {
            const resetTime = new Date(endOfDay);
            const hoursUntilReset = Math.ceil((resetTime.getTime() - Date.now()) / (1000 * 60 * 60));
            logger_1.default.warn(`Booking rate limit exceeded for IP ${clientIP}`, {
                successfulBookingsToday,
                limit: DAILY_BOOKING_LIMIT,
                hoursUntilReset,
            });
            return res.status(429).json({
                status: 'error',
                message: 'Daily booking limit exceeded. You can create maximum 5 bookings per day.',
                error: {
                    code: 'BOOKING_RATE_LIMIT_EXCEEDED',
                    limit: DAILY_BOOKING_LIMIT,
                    current: successfulBookingsToday,
                    resetTime: resetTime.toISOString(),
                    hoursUntilReset,
                },
            });
        }
        // Store client IP for later use
        req.clientIP = clientIP;
        next();
    }
    catch (error) {
        logger_1.default.error(`Error in booking rate limiting: ${error.message}`);
        // Don't block the request if rate limiting fails
        next();
    }
});
exports.bookingRateLimit = bookingRateLimit;
