"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.restrictTo = exports.protect = void 0;
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const errorHandler_1 = require("../utils/errorHandler");
const app_1 = __importDefault(require("../config/app"));
const models_1 = __importDefault(require("../models"));
// Protect routes - check if user is authenticated
exports.protect = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    console.log(' [AUTH] Starting authentication check...');
    console.log(' [AUTH] Request method:', req.method);
    console.log(' [AUTH] Request path:', req.path);
    console.log(' [AUTH] Authorization header:', req.headers.authorization ? 'Present' : 'Missing');
    // 1) Get token and check if it exists
    let token;
    if (req.headers.authorization && req.headers.authorization.startsWith('Bearer')) {
        token = req.headers.authorization.split(' ')[1];
        console.log(' [AUTH] Token extracted from Bearer header');
    }
    if (!token) {
        console.log(' [AUTH] No token provided - returning 401');
        return next(new errorHandler_1.AppError('You are not logged in! Please log in to get access.', 401));
    }
    // 2) Verify token
    console.log('🔐 [AUTH] Verifying JWT token...');
    try {
        const secret = Buffer.from(app_1.default.jwtSecret, 'utf-8');
        const decoded = jsonwebtoken_1.default.verify(token, secret);
        console.log('✅ [AUTH] Token verified successfully, user ID:', decoded.id);
        // 3) Check if user still exists
        console.log('🔐 [AUTH] Looking up user in database...');
        const currentUser = yield models_1.default.User.findByPk(decoded.id);
        if (!currentUser) {
            console.log('❌ [AUTH] User not found in database');
            return next(new errorHandler_1.AppError('The user belonging to this token no longer exists.', 401));
        }
        console.log('✅ [AUTH] User found:', { id: currentUser.id, email: currentUser.email, role: currentUser.role });
        // GRANT ACCESS TO PROTECTED ROUTE
        req.user = currentUser;
        console.log('✅ [AUTH] Authentication successful - proceeding to next middleware');
        next();
    }
    catch (error) {
        console.error('💥 [AUTH] JWT verification failed:', error);
        return next(new errorHandler_1.AppError('Invalid token. Please log in again.', 401));
    }
}));
// Restrict access to certain roles
const restrictTo = (...roles) => {
    return (req, res, next) => {
        if (!roles.includes(req.user.role)) {
            return next(new errorHandler_1.AppError('You do not have permission to perform this action', 403));
        }
        next();
    };
};
exports.restrictTo = restrictTo;
