"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = __importDefault(require("express"));
const morgan_1 = __importDefault(require("morgan"));
const cors_1 = __importDefault(require("cors"));
const helmet_1 = __importDefault(require("helmet"));
const compression_1 = __importDefault(require("compression"));
const path_1 = __importDefault(require("path"));
const models_1 = require("./models");
const app_1 = __importDefault(require("./config/app"));
const routes_1 = __importDefault(require("./routes"));
const errorHandler_1 = require("./utils/errorHandler");
const cleanupScheduler_1 = require("./utils/cleanupScheduler");
// Create Express app
const app = (0, express_1.default)();
// Set security HTTP headers
app.use((0, helmet_1.default)({
    contentSecurityPolicy: {
        directives: {
            defaultSrc: ["'self'"],
            styleSrc: ["'self'", "'unsafe-inline'"],
            scriptSrc: ["'self'"],
            imgSrc: ["'self'", "data:", "http://localhost:3000", "https://localhost:3000"],
            connectSrc: ["'self'"],
            fontSrc: ["'self'"],
            objectSrc: ["'none'"],
            mediaSrc: ["'self'"],
            frameSrc: ["'none'"],
        },
    },
}));
app.use((0, cors_1.default)({
    origin: '*',
    credentials: true,
}));
// Compression
app.use((0, compression_1.default)());
// Body parser
app.use(express_1.default.json({ limit: '10kb' }));
app.use(express_1.default.urlencoded({ extended: true, limit: '10kb' }));
// Serve static files with proper headers
app.use('/static', (req, res, next) => {
    // Set CORS headers for static files
    res.header('Access-Control-Allow-Origin', '*');
    res.header('Access-Control-Allow-Methods', 'GET');
    res.header('Access-Control-Allow-Headers', 'Content-Type');
    // Add Cross-Origin-Resource-Policy header to allow cross-origin embedding
    res.header('Cross-Origin-Resource-Policy', 'cross-origin');
    next();
}, express_1.default.static(path_1.default.join(__dirname, 'public')));
// Development logging
if (app_1.default.env === 'development') {
    app.use((0, morgan_1.default)('dev'));
}
// Rate limiting
// const limiter = rateLimit({
//   max: 100, // 100 requests per IP
//   windowMs: 60 * 60 * 1000, // 1 hour
//   message: 'Too many requests from this IP, please try again in an hour!',
// });
// app.use('/api', limiter);
// Health check endpoint
app.get('/health', (req, res) => {
    res.status(200).json({
        status: 'success',
        message: 'Server is healthy',
        environment: app_1.default.env,
        timestamp: new Date(),
    });
});
// API routes
app.use('/api/v1', routes_1.default);
// 404 handler
app.use(errorHandler_1.notFoundHandler);
// Global error handler
app.use(errorHandler_1.globalErrorHandler);
// Start server
const PORT = app_1.default.port;
app.listen(PORT, () => __awaiter(void 0, void 0, void 0, function* () {
    console.log(`Server running in ${app_1.default.env} mode on port ${PORT}`);
    // Test database connection
    try {
        yield models_1.sequelize.authenticate();
        console.log('Database connection established successfully.');
        // Start sync scheduler (calendar, reviews, and checkout feedback - run every 30 minutes by default)
        // startSync(); // DISABLED: Auto scheduled syncing deactivated
        // Start cleanup scheduler (runs daily at midnight to reset rate limits)
        (0, cleanupScheduler_1.startCleanupScheduler)();
    }
    catch (error) {
        console.error('Unable to connect to the database:', error);
    }
}));
// Handle unhandled promise rejections
process.on('unhandledRejection', (err) => {
    console.error('UNHANDLED REJECTION! 💥 Shutting down...');
    console.error(err.name, err.message);
    process.exit(1);
});
// Handle uncaught exceptions
process.on('uncaughtException', (err) => {
    console.error('UNCAUGHT EXCEPTION! 💥 Shutting down...');
    console.error(err.name, err.message);
    process.exit(1);
});
