"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteUser = exports.updateUser = exports.createUser = exports.getUserById = exports.getAllUsers = exports.updatePassword = exports.getCurrentUser = exports.login = void 0;
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const errorHandler_1 = require("../utils/errorHandler");
const passwordUtils_1 = require("../utils/passwordUtils");
const app_1 = __importDefault(require("../config/app"));
const models_1 = __importDefault(require("../models"));
// Generate JWT token
const signToken = (id) => {
    const payload = { id };
    const secret = Buffer.from(app_1.default.jwtSecret, 'utf-8');
    // Define options with correctly typed expiresIn
    const options = {
        expiresIn: app_1.default.jwtExpiresIn
    };
    return jsonwebtoken_1.default.sign(payload, secret, options);
};
// Create and send token response
const createSendToken = (user, statusCode, res) => {
    const token = signToken(user.id);
    // Remove password from output
    user.password = undefined;
    res.status(statusCode).json({
        status: 'success',
        token,
        data: {
            user,
        },
    });
};
// ================== AUTH CONTROLLER FUNCTIONS ==================
// Login user
exports.login = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { email, password } = req.body;
    // Check if email and password exist
    if (!email || !password) {
        return next(new errorHandler_1.AppError('Please provide email and password', 400));
    }
    // Check if user exists and password is correct
    const user = yield models_1.default.User.findOne({ where: { email } });
    if (!user) {
        return next(new errorHandler_1.AppError('Incorrect email or password', 401));
    }
    // Verify password
    const isPasswordCorrect = yield (0, passwordUtils_1.comparePassword)(password, user.password);
    if (!isPasswordCorrect) {
        return next(new errorHandler_1.AppError('Incorrect email or password', 401));
    }
    // If everything is ok, send token to client
    createSendToken(user, 200, res);
}));
// Get current user
exports.getCurrentUser = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const user = req.user;
    res.status(200).json({
        status: 'success',
        data: {
            user,
        },
    });
}));
// Update current user password
exports.updatePassword = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { currentPassword, newPassword, passwordConfirm } = req.body;
    // Check if passwords are provided
    if (!currentPassword || !newPassword || !passwordConfirm) {
        return next(new errorHandler_1.AppError('Please provide current password, new password and password confirmation', 400));
    }
    // Check if new password and confirmation match
    if (newPassword !== passwordConfirm) {
        return next(new errorHandler_1.AppError('New password and confirmation do not match', 400));
    }
    // Get current user
    const user = yield models_1.default.User.findByPk(req.params.id);
    // Check if current password is correct
    const isPasswordCorrect = yield (0, passwordUtils_1.comparePassword)(currentPassword, user.password);
    if (!isPasswordCorrect) {
        return next(new errorHandler_1.AppError('Your current password is incorrect', 401));
    }
    // Hash new password
    const hashedPassword = yield (0, passwordUtils_1.hashPassword)(newPassword);
    // Update password
    user.password = hashedPassword;
    yield user.save();
    // Log user in with new token
    createSendToken(user, 200, res);
}));
// ================== USER CONTROLLER FUNCTIONS ==================
// Get all users (admin only)
exports.getAllUsers = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const users = yield models_1.default.User.findAll({
        attributes: { exclude: ['password'] },
    });
    res.status(200).json({
        status: 'success',
        results: users.length,
        data: {
            users,
        },
    });
}));
// Get user by ID
exports.getUserById = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const user = yield models_1.default.User.findByPk(req.params.id, {
        attributes: { exclude: ['password'] },
    });
    if (!user) {
        return next(new errorHandler_1.AppError('No user found with that ID', 404));
    }
    res.status(200).json({
        status: 'success',
        data: {
            user,
        },
    });
}));
// Create a new user (admin only)
exports.createUser = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const newUser = yield models_1.default.User.create({
        firstName: req.body.firstName,
        lastName: req.body.lastName,
        email: req.body.email,
        password: req.body.password,
        role: req.body.role || 'staff',
    });
    // Remove password from output
    newUser.password = undefined;
    res.status(201).json({
        status: 'success',
        data: {
            user: newUser,
        },
    });
}));
// Update user
exports.updateUser = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    // Do not allow password updates via this route
    if (req.body.password) {
        return next(new errorHandler_1.AppError('This route is not for password updates. Please use /updatePassword.', 400));
    }
    const user = yield models_1.default.User.findByPk(req.params.id);
    if (!user) {
        return next(new errorHandler_1.AppError('No user found with that ID', 404));
    }
    // Update user fields
    yield user.update({
        firstName: req.body.firstName || user.firstName,
        lastName: req.body.lastName || user.lastName,
        email: req.body.email || user.email,
        role: req.body.role || user.role,
    });
    // Remove password from output
    user.password = undefined;
    res.status(200).json({
        status: 'success',
        data: {
            user,
        },
    });
}));
// Delete user
exports.deleteUser = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const user = yield models_1.default.User.findByPk(req.params.id);
    if (!user) {
        return next(new errorHandler_1.AppError('No user found with that ID', 404));
    }
    yield user.destroy();
    res.status(204).json({
        status: 'success',
        data: null,
    });
}));
