"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkRoomAvailability = exports.deleteRoom = exports.updateRoom = exports.createRoom = exports.deleteRoomImages = exports.uploadRoomImages = exports.getRoomById = exports.getAllRooms = void 0;
const sequelize_1 = require("sequelize");
const errorHandler_1 = require("../utils/errorHandler");
const models_1 = __importDefault(require("../models"));
const fileUpload_1 = __importStar(require("../utils/fileUpload"));
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
// Get all rooms with filters
exports.getAllRooms = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { status, roomType, capacity, priceMin, priceMax, floor } = req.query;
    const whereClause = {};
    // Apply filters if provided
    if (status)
        whereClause.status = status;
    if (roomType)
        whereClause.roomTypeId = roomType;
    if (capacity)
        whereClause.capacity = { [sequelize_1.Op.gte]: Number(capacity) };
    if (floor)
        whereClause.floor = floor;
    // Price range filter
    if (priceMin || priceMax) {
        whereClause.pricePerNight = {};
        if (priceMin)
            whereClause.pricePerNight[sequelize_1.Op.gte] = Number(priceMin);
        if (priceMax)
            whereClause.pricePerNight[sequelize_1.Op.lte] = Number(priceMax);
    }
    const rooms = yield models_1.default.Room.findAll({
        where: whereClause,
        include: [
            {
                model: models_1.default.RoomType,
                as: 'roomType',
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        results: rooms.length,
        data: {
            rooms,
        },
    });
}));
// Get room by ID
exports.getRoomById = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const room = yield models_1.default.Room.findByPk(req.params.id, {
        include: [
            {
                model: models_1.default.RoomType,
                as: 'roomType',
            },
        ],
    });
    if (!room) {
        return next(new errorHandler_1.AppError('No room found with that ID', 404));
    }
    res.status(200).json({
        status: 'success',
        data: {
            room,
        },
    });
}));
// Middleware for handling room image uploads
exports.uploadRoomImages = fileUpload_1.default.array('roomImages', 10); // Max 10 images per room
// Middleware for handling room image deletion
exports.deleteRoomImages = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    // Check if req.body exists and has imagesToDelete array
    if (req.body && req.body.imagesToDelete && Array.isArray(req.body.imagesToDelete)) {
        const imagesToDelete = req.body.imagesToDelete;
        // Delete files from file system
        imagesToDelete.forEach((filename) => {
            const fullPath = path_1.default.join(__dirname, '../public/uploads/rooms', filename);
            if (fs_1.default.existsSync(fullPath)) {
                try {
                    fs_1.default.unlinkSync(fullPath);
                }
                catch (error) {
                    console.error(`Failed to delete image file: ${fullPath}`, error);
                }
            }
        });
    }
    next();
}));
// Create a new room
exports.createRoom = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    // Check if room type exists
    const roomType = yield models_1.default.RoomType.findByPk(req.body.roomTypeId);
    if (!roomType) {
        return next(new errorHandler_1.AppError('Room type not found', 404));
    }
    // Process uploaded files if any
    const images = [];
    if (req.files && Array.isArray(req.files)) {
        const files = req.files;
        files.forEach(file => {
            // Store only filename in the database
            const filename = (0, fileUpload_1.getFilename)(file.path);
            images.push(filename);
        });
    }
    // Create new room
    const newRoom = yield models_1.default.Room.create({
        roomTypeId: req.body.roomTypeId,
        number: req.body.number,
        floor: req.body.floor,
        capacity: req.body.capacity,
        pricePerNight: req.body.pricePerNight || roomType.basePrice,
        status: req.body.status || 'available',
        description: req.body.description,
        images: images,
        discount: req.body.discount,
    });
    res.status(201).json({
        status: 'success',
        data: {
            room: newRoom,
        },
    });
}));
// Update room
exports.updateRoom = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    const room = yield models_1.default.Room.findByPk(req.params.id);
    if (!room) {
        return next(new errorHandler_1.AppError('No room found with that ID', 404));
    }
    // If room type is being updated, verify it exists
    if (req.body && req.body.roomTypeId) {
        const roomType = yield models_1.default.RoomType.findByPk(req.body.roomTypeId);
        if (!roomType) {
            return next(new errorHandler_1.AppError('Room type not found', 404));
        }
    }
    // Get current images (raw filenames, not transformed paths)
    console.log('🔍 Debugging room images:');
    console.log('room.images (transformed):', room.images);
    console.log('room.getDataValue("images"):', room.getDataValue('images'));
    console.log('room.dataValues.images:', (_a = room.dataValues) === null || _a === void 0 ? void 0 : _a.images);
    // Get raw images data and ensure it's an array
    let currentImages = room.getDataValue('images') || [];
    // If it's a string, parse it
    if (typeof currentImages === 'string') {
        try {
            currentImages = JSON.parse(currentImages);
        }
        catch (e) {
            console.error("Error parsing images JSON, starting fresh.", e);
            currentImages = [];
        }
    }
    // Ensure it's an array, even after parsing
    if (!Array.isArray(currentImages)) {
        currentImages = [];
    }
    console.log('✅ Final currentImages:', currentImages);
    // Handle image deletion - remove from database list
    if (req.body && req.body.imagesToDelete && Array.isArray(req.body.imagesToDelete)) {
        const imagesToDelete = req.body.imagesToDelete;
        currentImages = currentImages.filter((img) => !imagesToDelete.includes(img));
    }
    // Handle new image uploads - add to existing images
    if (req.files && Array.isArray(req.files)) {
        const files = req.files;
        files.forEach(file => {
            // Store only filename in the database
            const filename = (0, fileUpload_1.getFilename)(file.path);
            currentImages.push(filename);
        });
    }
    console.log(currentImages);
    // Update room fields
    yield room.update({
        roomTypeId: (req.body && req.body.roomTypeId) || room.roomTypeId,
        number: (req.body && req.body.number) || room.number,
        floor: (req.body && req.body.floor) || room.floor,
        capacity: (req.body && req.body.capacity) || room.capacity,
        pricePerNight: (req.body && req.body.pricePerNight) || room.pricePerNight,
        status: (req.body && req.body.status) || room.status,
        description: (req.body && req.body.description) || room.description,
        images: currentImages,
        discount: (req.body && req.body.discount) || room.discount,
    });
    res.status(200).json({
        status: 'success',
        data: {
            room,
        },
    });
}));
// Delete room
exports.deleteRoom = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const room = yield models_1.default.Room.findByPk(req.params.id);
    if (!room) {
        return next(new errorHandler_1.AppError('No room found with that ID', 404));
    }
    // Check if room is associated with any bookings
    const bookings = yield models_1.default.Booking.findOne({
        where: {
            roomId: room.id,
            status: {
                [sequelize_1.Op.notIn]: ['cancelled', 'completed'],
            },
        },
    });
    if (bookings) {
        return next(new errorHandler_1.AppError('Cannot delete room with active bookings', 400));
    }
    // Delete associated image files if they exist
    const imagesRaw = room.getDataValue('images');
    if (imagesRaw) {
        let imageFilenames = [];
        if (typeof imagesRaw === 'string') {
            try {
                imageFilenames = JSON.parse(imagesRaw);
            }
            catch (error) {
                console.error('Error parsing room images JSON:', error);
                imageFilenames = []; // Fallback to empty array
            }
        }
        else if (Array.isArray(imagesRaw)) {
            imageFilenames = imagesRaw;
        }
        if (imageFilenames.length > 0) {
            imageFilenames.forEach((filename) => {
                const fullPath = path_1.default.join(__dirname, '../public/uploads/rooms', filename);
                if (fs_1.default.existsSync(fullPath)) {
                    try {
                        fs_1.default.unlinkSync(fullPath);
                    }
                    catch (err) {
                        console.error(`Failed to delete file: ${fullPath}`, err);
                    }
                }
            });
        }
    }
    yield room.destroy();
    res.status(204).json({
        status: 'success',
        data: null,
    });
}));
// Check room availability for a date range
exports.checkRoomAvailability = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    // Changed from req.body to req.query for GET request
    const { checkInDate, checkOutDate } = req.query;
    if (!checkInDate || !checkOutDate) {
        return next(new errorHandler_1.AppError('Please provide check-in and check-out dates', 400));
    }
    // Parse dates
    const [day, month, year] = checkInDate.split('/');
    const [outDay, outMonth, outYear] = checkOutDate.split('/');
    const checkIn = new Date(+year, +month - 1, +day);
    const checkOut = new Date(+outYear, +outMonth - 1, +outDay);
    // Validate dates
    if (isNaN(checkIn.getTime()) || isNaN(checkOut.getTime())) {
        return next(new errorHandler_1.AppError('Invalid date format', 400));
    }
    if (checkIn >= checkOut) {
        return next(new errorHandler_1.AppError('Check-in date must be before check-out date', 400));
    }
    // Get all rooms that are available for the entire date range
    const unavailableRoomIds = yield models_1.default.Booking.findAll({
        attributes: ['roomId'],
        where: {
            status: {
                [sequelize_1.Op.notIn]: ['cancelled'],
            },
            [sequelize_1.Op.or]: [
                {
                    // Bookings that start during the requested period (excluding checkout date)
                    checkInDate: {
                        [sequelize_1.Op.gte]: checkIn,
                        [sequelize_1.Op.lt]: checkOut,
                    },
                },
                {
                    // Bookings that end during the requested period (excluding checkin date)
                    checkOutDate: {
                        [sequelize_1.Op.gt]: checkIn,
                        [sequelize_1.Op.lte]: checkOut,
                    },
                },
                {
                    // Bookings that span the entire requested period
                    [sequelize_1.Op.and]: [
                        {
                            checkInDate: {
                                [sequelize_1.Op.lt]: checkIn,
                            },
                        },
                        {
                            checkOutDate: {
                                [sequelize_1.Op.gt]: checkOut,
                            },
                        },
                    ],
                },
            ],
        },
        raw: true,
    });
    // Extract room IDs from the result
    const unavailableIds = unavailableRoomIds.map((booking) => booking.roomId);
    // Find all rooms that are not in the unavailable list and not in maintenance
    const availableRooms = yield models_1.default.Room.findAll({
        where: {
            id: {
                [sequelize_1.Op.notIn]: unavailableIds,
            },
            status: {
                [sequelize_1.Op.notIn]: ['maintenance'],
            },
        },
        include: [
            {
                model: models_1.default.RoomType,
                as: 'roomType',
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        results: availableRooms.length,
        data: {
            availableRooms,
        },
    });
}));
