"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateReviewVisibility = exports.syncReviews = exports.getReviewStats = exports.deleteReview = exports.updateReview = exports.createReview = exports.getReviewById = exports.getAllReviews = exports.getRoomReviews = exports.getPublicReviews = void 0;
const sequelize_1 = require("sequelize");
const models_1 = __importDefault(require("../models"));
const reviewSync_1 = require("../utils/reviewSync");
// Public routes
const getPublicReviews = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { source, rating, limit = 10, offset = 0 } = req.query;
        const where = { isPublished: true };
        if (source)
            where.source = source;
        if (rating)
            where.rating = rating;
        const reviews = yield models_1.default.Review.findAndCountAll({
            where,
            limit: Number(limit),
            offset: Number(offset),
            order: [['createdAt', 'DESC']],
            include: [{
                    model: models_1.default.Room,
                    as: 'room',
                    attributes: ['id', 'number', 'floor']
                }]
        });
        return res.status(200).json({
            status: 'success',
            data: {
                reviews: reviews.rows,
                total: reviews.count,
                limit: Number(limit),
                offset: Number(offset)
            }
        });
    }
    catch (error) {
        console.error('Error getting public reviews:', error);
        return res.status(500).json({ message: 'Failed to retrieve reviews', error: error.message });
    }
});
exports.getPublicReviews = getPublicReviews;
const getRoomReviews = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId, id } = req.params;
        const actualRoomId = roomId || id; // Handle both /reviews/room/:roomId and /rooms/:id/reviews
        const { source, rating, limit = 10, offset = 0 } = req.query;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(actualRoomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        const where = { roomId: actualRoomId, isPublished: true };
        if (source)
            where.source = source;
        if (rating)
            where.rating = rating;
        const reviews = yield models_1.default.Review.findAll({
            where,
            limit: Number(limit),
            offset: Number(offset),
            order: [['createdAt', 'DESC']],
            include: [{
                    model: models_1.default.Guest,
                    as: 'guest',
                    attributes: ['id', 'firstName', 'lastName']
                }]
        });
        return res.status(200).json({
            status: 'success',
            data: {
                reviews: reviews,
                total: reviews.length,
                limit: Number(limit),
                offset: Number(offset)
            }
        });
    }
    catch (error) {
        console.error('Error getting room reviews:', error);
        return res.status(500).json({ message: 'Failed to retrieve room reviews', error: error.message });
    }
});
exports.getRoomReviews = getRoomReviews;
// Protected routes
const getAllReviews = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { source, rating, isPublished, limit = 10, offset = 0 } = req.query;
        const where = {};
        if (source)
            where.source = source;
        if (rating)
            where.rating = rating;
        if (isPublished !== undefined)
            where.isPublished = isPublished;
        const reviews = yield models_1.default.Review.findAll({
            where,
            limit: Number(limit),
            offset: Number(offset),
            order: [['createdAt', 'DESC']],
            include: [
                {
                    model: models_1.default.Room,
                    as: 'room',
                    attributes: ['id', 'number', 'floor']
                },
                {
                    model: models_1.default.Guest,
                    as: 'guest',
                    attributes: ['id', 'firstName', 'lastName', 'email']
                }
            ]
        });
        return res.status(200).json({
            status: 'success',
            data: {
                reviews: reviews,
                total: reviews.length,
                limit: Number(limit),
                offset: Number(offset)
            }
        });
    }
    catch (error) {
        console.error('Error getting all reviews:', error);
        return res.status(500).json({ message: 'Failed to retrieve reviews', error: error.message });
    }
});
exports.getAllReviews = getAllReviews;
const getReviewById = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        const review = yield models_1.default.Review.findByPk(id, {
            include: [
                {
                    model: models_1.default.Room,
                    as: 'room',
                    attributes: ['id', 'number', 'floor']
                },
                {
                    model: models_1.default.Guest,
                    as: 'guest',
                    attributes: ['id', 'firstName', 'lastName', 'email']
                }
            ]
        });
        if (!review) {
            return res.status(404).json({ message: 'Review not found' });
        }
        return res.status(200).json({
            status: 'success',
            data: { review }
        });
    }
    catch (error) {
        console.error('Error getting review:', error);
        return res.status(500).json({ message: 'Failed to retrieve review', error: error.message });
    }
});
exports.getReviewById = getReviewById;
const createReview = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId, guestId, rating, comment } = req.body;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        // Check if guest exists
        const guest = yield models_1.default.Guest.findByPk(guestId);
        if (!guest) {
            return res.status(404).json({ message: 'Guest not found' });
        }
        // Check if guest has already reviewed this room
        const existingReview = yield models_1.default.Review.findOne({
            where: { roomId, guestId }
        });
        if (existingReview) {
            return res.status(400).json({ message: 'Guest has already reviewed this room' });
        }
        const review = yield models_1.default.Review.create({
            roomId,
            guestId,
            rating,
            comment,
            isPublished: true
        });
        return res.status(201).json({
            status: 'success',
            data: { review }
        });
    }
    catch (error) {
        console.error('Error creating review:', error);
        return res.status(500).json({ message: 'Failed to create review', error: error.message });
    }
});
exports.createReview = createReview;
const updateReview = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        const { rating, comment, isPublished } = req.body;
        const review = yield models_1.default.Review.findByPk(id);
        if (!review) {
            return res.status(404).json({ message: 'Review not found' });
        }
        yield review.update({
            rating: rating !== undefined ? rating : review.rating,
            comment: comment !== undefined ? comment : review.comment,
            isPublished: isPublished !== undefined ? isPublished : review.isPublished
        });
        return res.status(200).json({
            status: 'success',
            data: { review }
        });
    }
    catch (error) {
        console.error('Error updating review:', error);
        return res.status(500).json({ message: 'Failed to update review', error: error.message });
    }
});
exports.updateReview = updateReview;
const deleteReview = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        const review = yield models_1.default.Review.findByPk(id);
        if (!review) {
            return res.status(404).json({ message: 'Review not found' });
        }
        yield review.destroy();
        return res.status(200).json({
            status: 'success',
            message: 'Review deleted successfully'
        });
    }
    catch (error) {
        console.error('Error deleting review:', error);
        return res.status(500).json({ message: 'Failed to delete review', error: error.message });
    }
});
exports.deleteReview = deleteReview;
/**
 * Get review statistics for a room
 * @route GET /api/rooms/:roomId/reviews/stats
 */
const getReviewStats = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        // Get review statistics
        const stats = yield models_1.default.Review.findAll({
            where: {
                roomId,
                isPublished: true,
            },
            attributes: [
                [(0, sequelize_1.fn)('COUNT', (0, sequelize_1.col)('id')), 'totalReviews'],
                [(0, sequelize_1.fn)('AVG', (0, sequelize_1.col)('rating')), 'averageRating'],
                [(0, sequelize_1.fn)('MIN', (0, sequelize_1.col)('rating')), 'minRating'],
                [(0, sequelize_1.fn)('MAX', (0, sequelize_1.col)('rating')), 'maxRating'],
            ],
        });
        return res.status(200).json({
            status: 'success',
            data: {
                stats: stats[0],
            },
        });
    }
    catch (error) {
        console.error('Error getting review statistics:', error);
        return res.status(500).json({ message: 'Failed to retrieve review statistics', error: error.message });
    }
});
exports.getReviewStats = getReviewStats;
/**
 * Sync external reviews for a room
 * @route POST /api/rooms/:roomId/reviews/sync
 */
const syncReviews = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        const { platform } = req.query;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        yield (0, reviewSync_1.syncExternalReviews)(parseInt(roomId), platform);
        return res.status(200).json({
            status: 'success',
            message: 'Reviews synced successfully',
        });
    }
    catch (error) {
        console.error('Error syncing reviews:', error);
        return res.status(500).json({ message: 'Failed to sync reviews', error: error.message });
    }
});
exports.syncReviews = syncReviews;
/**
 * Update review visibility
 * @route PATCH /api/reviews/:id/visibility
 */
const updateReviewVisibility = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        const { isPublished } = req.body;
        // Find the review
        const review = yield models_1.default.Review.findByPk(id);
        if (!review) {
            return res.status(404).json({ message: 'Review not found' });
        }
        // Update review visibility
        yield review.update({
            isPublished,
        });
        return res.status(200).json({
            status: 'success',
            data: {
                review,
            },
        });
    }
    catch (error) {
        console.error('Error updating review visibility:', error);
        return res.status(500).json({ message: 'Failed to update review visibility', error: error.message });
    }
});
exports.updateReviewVisibility = updateReviewVisibility;
