"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.processWebhook = void 0;
const errorHandler_1 = require("../utils/errorHandler");
const konnectPayment_service_1 = require("../services/konnectPayment.service");
const models_1 = __importDefault(require("../models"));
const logger_1 = __importDefault(require("../utils/logger"));
/**
 * Handle Konnect payment webhook notifications
 * This endpoint will be called by Konnect when payment status changes
 */
exports.processWebhook = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { payment_ref } = req.query;
        const userAgent = req.get('User-Agent') || '';
        const isUserRedirect = userAgent.includes('Mozilla') || userAgent.includes('Chrome') || userAgent.includes('Safari') || userAgent.includes('Edge');
        // Use a reliable frontend URL from environment variables
        const frontendUrl = process.env.FRONTEND_URL;
        if (!frontendUrl) {
            logger_1.default.error('FRONTEND_URL environment variable is not set. This is required for user redirects.');
            // For server-to-server webhooks, we can continue, but user redirects will fail.
            if (isUserRedirect) {
                return res.status(500).send('Server configuration error: Frontend URL not specified.');
            }
        }
        logger_1.default.info(`Received Konnect ${isUserRedirect ? 'user redirect' : 'webhook'} for payment_ref: ${payment_ref}`, {
            userAgent,
            isUserRedirect,
            frontendUrl: frontendUrl
        });
        if (!payment_ref) {
            logger_1.default.error('Webhook called without payment_ref parameter');
            if (isUserRedirect) {
                // Redirect user to frontend error page
                return res.redirect(`${frontendUrl}/payment/error?message=Invalid payment reference`);
            }
            return res.status(400).json({
                status: 'error',
                message: 'Missing payment_ref parameter',
            });
        }
        // Process the webhook notification
        const updatedPayment = yield (0, konnectPayment_service_1.processWebhook)(payment_ref);
        if (!updatedPayment) {
            logger_1.default.error(`Payment not found for payment_ref: ${payment_ref}`);
            if (isUserRedirect) {
                // Redirect user to frontend error page
                return res.redirect(`${frontendUrl}/payment/error?message=Payment not found`);
            }
            return res.status(404).json({
                status: 'error',
                message: 'Payment not found',
            });
        }
        // If payment failed or expired, delete the associated booking
        if (updatedPayment.status === 'failed') {
            const booking = yield models_1.default.Booking.findByPk(updatedPayment.bookingId);
            if (booking && booking.status !== 'cancelled') {
                const bookingId = booking.id;
                yield booking.destroy();
                logger_1.default.info(`Deleted booking ${bookingId} due to failed/expired payment ${payment_ref}`, {
                    paymentId: updatedPayment.id,
                    bookingId: bookingId
                });
            }
        }
        // If this is a user redirect, send them to the appropriate frontend page
        if (isUserRedirect) {
            const bookingId = updatedPayment.bookingId || 'unknown';
            if (updatedPayment.status === 'completed') {
                logger_1.default.info(`Redirecting user to success page for payment ${payment_ref}`, { bookingId });
                return res.redirect(`${frontendUrl}/booking/success?booking_id=${bookingId}&payment_ref=${payment_ref}`);
            }
            else if (updatedPayment.status === 'pending') {
                logger_1.default.info(`Redirecting user to pending page for payment ${payment_ref}`, { bookingId });
                return res.redirect(`${frontendUrl}/payment/pending?booking_id=${bookingId}&payment_ref=${payment_ref}`);
            }
            else {
                // For failed or expired statuses, redirect to failure page
                logger_1.default.info(`Redirecting user to failure page for payment ${payment_ref} (status: ${updatedPayment.status})`, {
                    bookingId,
                    bookingDeleted: updatedPayment.status === 'failed'
                });
                return res.redirect(`${frontendUrl}/payment/failed?booking_id=${bookingId}&payment_ref=${payment_ref}`);
            }
        }
        // For server-to-server webhooks, just send a success response
        logger_1.default.info(`Webhook for payment ${payment_ref} processed successfully with status: ${updatedPayment.status}`);
        // Server-to-server webhook response
        res.status(200).json({
            status: 'success',
            message: 'Webhook processed successfully',
        });
    }
    catch (error) {
        logger_1.default.error('Error processing Konnect webhook:', error);
        return res.status(500).json({
            status: 'error',
            message: 'Failed to process webhook',
        });
    }
}));
