"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPaymentStatus = exports.verifyPaymentAccess = void 0;
const errorHandler_1 = require("../utils/errorHandler");
const models_1 = __importDefault(require("../models"));
const logger_1 = __importDefault(require("../utils/logger"));
// Normalize IP address for consistent storage and comparison
const normalizeIP = (ip) => {
    if (!ip)
        return '127.0.0.1';
    // Clean up IPv6 mapped IPv4 addresses
    if (ip.startsWith('::ffff:')) {
        ip = ip.substring(7);
    }
    // Convert IPv6 localhost to IPv4 localhost for consistency
    if (ip === '::1') {
        return '127.0.0.1';
    }
    return ip;
};
/**
 * Verify payment access for frontend pages using IP-based validation
 * This endpoint validates that a user has legitimate access to view payment results
 */
exports.verifyPaymentAccess = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e, _f;
    const { booking_id, payment_ref, user_ip } = req.query;
    const rawClientIP = req.clientIP || req.ip || '127.0.0.1';
    const requestingIP = normalizeIP(rawClientIP);
    // Use the original user IP from the URL parameter if provided, otherwise fall back to requesting IP
    const clientIP = user_ip ? normalizeIP(user_ip) : requestingIP;
    // Debug logging
    logger_1.default.info('Payment verification request received', {
        booking_id,
        payment_ref,
        user_ip,
        rawClientIP,
        requestingIP,
        clientIP,
        allQueryParams: req.query
    });
    if (!booking_id || !payment_ref) {
        return next(new errorHandler_1.AppError('Missing booking ID or payment reference', 400));
    }
    try {
        // Find the booking
        const booking = yield models_1.default.Booking.findByPk(booking_id, {
            include: [
                { model: models_1.default.Guest, as: 'guest' },
                { model: models_1.default.Room, as: 'room' }
            ]
        });
        if (!booking) {
            logger_1.default.warn(`Payment verification failed: Booking ${booking_id} not found`);
            return next(new errorHandler_1.AppError('Invalid booking reference', 404));
        }
        // Find the payment with the matching payment reference
        const payment = yield models_1.default.Payment.findOne({
            where: {
                bookingId: booking_id,
                transactionId: payment_ref
            }
        });
        if (!payment) {
            logger_1.default.warn(`Payment verification failed: Payment ${payment_ref} not found for booking ${booking_id}`);
            return next(new errorHandler_1.AppError('Invalid payment reference', 404));
        }
        // Check if payment was created recently (within last 2 hours for better UX)
        const twoHoursAgo = new Date(Date.now() - 2 * 60 * 60 * 1000);
        if (payment.createdAt < twoHoursAgo) {
            logger_1.default.warn(`Payment verification failed: Payment ${payment_ref} is older than 2 hours`);
            return next(new errorHandler_1.AppError('Payment reference has expired', 403));
        }
        // Check if this IP has a recent successful payment attempt for this payment
        logger_1.default.info('Looking up PaymentAttempt record', {
            paymentId: payment.id,
            clientIP,
            searchCriteria: {
                paymentId: payment.id,
                ipAddress: clientIP,
                isSuccessful: true
            }
        });
        const recentPaymentAttempt = yield models_1.default.PaymentAttempt.findOne({
            where: {
                paymentId: payment.id,
                ipAddress: clientIP,
                isSuccessful: true
            },
            order: [['createdAt', 'DESC']]
        });
        // Debug: Show all payment attempts for this payment
        const allPaymentAttempts = yield models_1.default.PaymentAttempt.findAll({
            where: {
                paymentId: payment.id
            },
            order: [['createdAt', 'DESC']]
        });
        logger_1.default.info('All PaymentAttempt records for this payment', {
            paymentId: payment.id,
            totalAttempts: allPaymentAttempts.length,
            attempts: allPaymentAttempts.map(attempt => ({
                id: attempt.id,
                ipAddress: attempt.ipAddress,
                isSuccessful: attempt.isSuccessful,
                createdAt: attempt.createdAt
            }))
        });
        if (!recentPaymentAttempt) {
            logger_1.default.warn(`Payment access denied: No successful payment attempt found for IP ${clientIP}`, {
                bookingId: booking_id,
                paymentRef: payment_ref,
                clientIP,
                searchedIP: clientIP,
                availableIPs: allPaymentAttempts.map(a => a.ipAddress)
            });
            return next(new errorHandler_1.AppError('Unauthorized access - payment not initiated from this location', 403));
        }
        // Log successful verification
        logger_1.default.info(`Payment access granted for booking ${booking_id}`, {
            bookingId: booking.id,
            paymentId: payment.id,
            paymentStatus: payment.status,
            guestEmail: (_a = booking.guest) === null || _a === void 0 ? void 0 : _a.email,
            clientIP
        });
        // Return verified payment and booking details
        res.status(200).json({
            status: 'success',
            data: {
                booking: {
                    id: booking.id,
                    status: booking.status,
                    totalPrice: booking.totalPrice,
                    checkInDate: booking.checkInDate,
                    checkOutDate: booking.checkOutDate,
                    guest: {
                        firstName: (_b = booking.guest) === null || _b === void 0 ? void 0 : _b.firstName,
                        lastName: (_c = booking.guest) === null || _c === void 0 ? void 0 : _c.lastName,
                        email: (_d = booking.guest) === null || _d === void 0 ? void 0 : _d.email
                    },
                    room: {
                        name: (_e = booking.room) === null || _e === void 0 ? void 0 : _e.name,
                        number: (_f = booking.room) === null || _f === void 0 ? void 0 : _f.number
                    }
                },
                payment: {
                    id: payment.id,
                    amount: payment.amount,
                    status: payment.status,
                    paymentMethod: payment.paymentMethod,
                    transactionId: payment.transactionId,
                    createdAt: payment.createdAt
                }
            }
        });
    }
    catch (error) {
        logger_1.default.error(`Error verifying payment access: ${error.message}`, {
            bookingId: booking_id,
            paymentRef: payment_ref,
            clientIP
        });
        return next(new errorHandler_1.AppError('Failed to verify payment access', 500));
    }
}));
/**
 * Get payment status for real-time updates
 */
exports.getPaymentStatus = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    const { payment_ref } = req.params;
    if (!payment_ref) {
        return next(new errorHandler_1.AppError('Missing payment reference', 400));
    }
    try {
        const payment = yield models_1.default.Payment.findOne({
            where: { transactionId: payment_ref },
            include: [
                {
                    model: models_1.default.Booking,
                    as: 'booking',
                    include: [
                        { model: models_1.default.Guest, as: 'guest' },
                        { model: models_1.default.Room, as: 'room' }
                    ]
                }
            ]
        });
        if (!payment) {
            return next(new errorHandler_1.AppError('Payment not found', 404));
        }
        res.status(200).json({
            status: 'success',
            data: {
                paymentStatus: payment.status,
                bookingId: (_a = payment.booking) === null || _a === void 0 ? void 0 : _a.id,
                lastUpdated: payment.updatedAt
            }
        });
    }
    catch (error) {
        logger_1.default.error(`Error getting payment status: ${error.message}`, { paymentRef: payment_ref });
        return next(new errorHandler_1.AppError('Failed to get payment status', 500));
    }
}));
