"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPaymentStatusByRef = exports.deletePayment = exports.updatePaymentStatus = exports.createPayment = exports.createManualPayment = exports.getPaymentById = exports.getAllPayments = void 0;
const errorHandler_1 = require("../utils/errorHandler");
const errorHandler_2 = require("../utils/errorHandler");
const models_1 = __importDefault(require("../models"));
const konnectPayment_service_1 = require("../services/konnectPayment.service");
const paymentRateLimit_middleware_1 = require("../middlewares/paymentRateLimit.middleware");
const logger_1 = __importDefault(require("../utils/logger"));
// Normalize IP address for consistent storage and comparison
const normalizeIP = (ip) => {
    if (!ip)
        return '127.0.0.1';
    // Clean up IPv6 mapped IPv4 addresses
    if (ip.startsWith('::ffff:')) {
        ip = ip.substring(7);
    }
    // Convert IPv6 localhost to IPv4 localhost for consistency
    if (ip === '::1') {
        return '127.0.0.1';
    }
    return ip;
};
// Get all payments
exports.getAllPayments = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { bookingId, status } = req.query;
    const whereClause = {};
    if (bookingId)
        whereClause.bookingId = bookingId;
    if (status)
        whereClause.status = status;
    const payments = yield models_1.default.Payment.findAll({
        where: whereClause,
        include: [
            {
                model: models_1.default.Booking,
                as: 'booking',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                    },
                    {
                        model: models_1.default.Guest,
                        as: 'guest',
                    },
                ],
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        results: payments.length,
        data: {
            payments,
        },
    });
}));
// Get payment by ID
exports.getPaymentById = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const payment = yield models_1.default.Payment.findByPk(req.params.id, {
        include: [
            {
                model: models_1.default.Booking,
                as: 'booking',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                    },
                    {
                        model: models_1.default.Guest,
                        as: 'guest',
                    },
                ],
            },
        ],
    });
    if (!payment) {
        return next(new errorHandler_2.AppError('No payment found with that ID', 404));
    }
    res.status(200).json({
        status: 'success',
        data: {
            payment,
        },
    });
}));
// Create a manual payment (admin/manager only)
// This registers an in-person or offline payment without initiating Konnect flow
exports.createManualPayment = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { bookingId, amount, paymentMethod, notes, status } = req.body;
    // Validate booking exists
    const booking = yield models_1.default.Booking.findByPk(bookingId, {
        include: [
            { model: models_1.default.Guest, as: 'guest' },
            { model: models_1.default.Room, as: 'room' },
        ],
    });
    if (!booking) {
        return next(new errorHandler_2.AppError('No booking found with that ID', 404));
    }
    if (!amount || amount <= 0) {
        return next(new errorHandler_2.AppError('Amount must be a positive number', 400));
    }
    // Create payment record directly
    const trxId = `MANUAL-${Date.now()}`;
    const payment = yield models_1.default.Payment.create({
        bookingId,
        amount,
        paymentMethod,
        status: status || 'completed',
        transactionId: trxId,
        notes,
    });
    // If booking is still pending and payment succeeded, confirm it
    if (booking.status === 'pending' && (status || 'completed') === 'completed') {
        yield booking.update({ status: 'confirmed' });
    }
    const created = yield models_1.default.Payment.findByPk(payment.id, {
        include: [
            { model: models_1.default.Booking, as: 'booking' },
        ],
    });
    res.status(201).json({
        status: 'success',
        data: {
            payment: created,
        },
    });
}));
// Create a new payment
exports.createPayment = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c, _d, _e;
    const { bookingId, paymentMethod, notes } = req.body;
    const rawClientIP = req.clientIP || req.ip || '127.0.0.1';
    const clientIP = normalizeIP(rawClientIP);
    const userAgent = req.get('User-Agent') || '';
    // Check if booking exists
    const booking = yield models_1.default.Booking.findByPk(bookingId, {
        include: [
            {
                model: models_1.default.Guest,
                as: 'guest',
            },
            {
                model: models_1.default.Room,
                as: 'room',
            },
        ],
    });
    if (!booking) {
        return next(new errorHandler_2.AppError('No booking found with that ID', 404));
    }
    if (!booking.totalPrice) {
        return next(new errorHandler_2.AppError('Could not process payment, total price not found for this booking.', 400));
    }
    // Check if there's already a successful or pending payment for this booking
    const existingPayment = yield models_1.default.Payment.findOne({
        where: {
            bookingId,
            status: ['completed', 'pending']
        }
    });
    if (existingPayment) {
        return next(new errorHandler_2.AppError('Payment already exists for this booking. Cannot create duplicate payment.', 400));
    }
    const amount = booking.totalPrice;
    // Process payment through Konnect payment service
    try {
        logger_1.default.info(`Creating payment for booking ${bookingId}`, { amount, paymentMethod });
        // Initiate payment with Konnect
        const customerDetails = {
            firstName: ((_a = booking.guest) === null || _a === void 0 ? void 0 : _a.firstName) || 'Guest',
            lastName: ((_b = booking.guest) === null || _b === void 0 ? void 0 : _b.lastName) || 'User',
            phoneNumber: ((_c = booking.guest) === null || _c === void 0 ? void 0 : _c.phone) || '',
            email: ((_d = booking.guest) === null || _d === void 0 ? void 0 : _d.email) || '',
        };
        const orderDetails = {
            orderId: booking.id.toString(),
            description: `Payment for booking at ${((_e = booking.room) === null || _e === void 0 ? void 0 : _e.name) || 'property'}`,
        };
        const paymentResult = yield (0, konnectPayment_service_1.initiatePayment)(amount, paymentMethod, customerDetails, orderDetails);
        logger_1.default.info(`Konnect payment initiated for booking ${bookingId}`, {
            transactionId: paymentResult.paymentRef
        });
        // Create payment record in database
        const payment = yield models_1.default.Payment.create({
            bookingId,
            amount,
            paymentMethod,
            status: 'pending',
            transactionId: paymentResult.paymentRef,
            paymentUrl: paymentResult.payUrl,
            notes,
        });
        // If booking is pending and payment successful, update booking status
        if (booking.status === 'pending') {
            yield booking.update({ status: 'confirmed' });
        }
        // Record successful payment attempt
        yield (0, paymentRateLimit_middleware_1.recordPaymentAttempt)(clientIP, true, payment.id, userAgent);
        // Get payment with relations
        const newPayment = yield models_1.default.Payment.findByPk(payment.id, {
            include: [
                {
                    model: models_1.default.Booking,
                    as: 'booking',
                },
            ],
        });
        res.status(201).json({
            status: 'success',
            data: {
                payment: newPayment,
                paymentUrl: paymentResult.payUrl,
            },
        });
    }
    catch (error) {
        // Type guard for the error object
        const customError = error;
        // Create failed payment record
        const failedPayment = yield models_1.default.Payment.create({
            bookingId,
            amount,
            paymentMethod,
            status: 'failed',
            notes: `${notes ? notes + ' - ' : ''}Failed with error: ${customError.message}`,
        });
        // Record failed payment attempt
        yield (0, paymentRateLimit_middleware_1.recordPaymentAttempt)(clientIP, false, failedPayment.id, userAgent);
        return next(new errorHandler_2.AppError(`Payment processing failed: ${customError.message}`, 400));
    }
}));
// Update payment status
exports.updatePaymentStatus = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { status, notes } = req.body;
    const payment = yield models_1.default.Payment.findByPk(req.params.id);
    if (!payment) {
        return next(new errorHandler_2.AppError('No payment found with that ID', 404));
    }
    // Handle refund
    if (status === 'refunded' && payment.status === 'completed') {
        try {
            logger_1.default.info(`Processing refund for payment ${payment.id}`, {
                transactionId: payment.transactionId,
                amount: payment.amount
            });
            // Process refund through Konnect payment service
            const refundResult = yield (0, konnectPayment_service_1.refundPayment)(payment.transactionId, payment.amount);
            logger_1.default.info(`Konnect refund processed for payment ${payment.id}`, {
                refundId: refundResult.refund.id,
                refundStatus: refundResult.refund.status
            });
            // Update payment record
            yield payment.update({
                status: refundResult.refund.status,
                notes: notes ? `${payment.notes ? payment.notes + ' - ' : ''}${notes} - Refund ID: ${refundResult.refund.id}` : `${payment.notes ? payment.notes + ' - ' : ''}Refunded. Refund ID: ${refundResult.refund.id}`,
            });
        }
        catch (error) {
            // Type guard for the error object
            const customError = error;
            logger_1.default.error(`Failed to process refund for payment ${payment.id}: ${customError.message}`);
            return next(new errorHandler_2.AppError(`Refund processing failed: ${customError.message}`, 400));
        }
    }
    else {
        // For other status updates
        yield payment.update({
            status,
            notes: notes ? `${payment.notes ? payment.notes + ' - ' : ''}${notes}` : payment.notes,
        });
    }
    // Get updated payment with relations
    const updatedPayment = yield models_1.default.Payment.findByPk(req.params.id, {
        include: [
            {
                model: models_1.default.Booking,
                as: 'booking',
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        data: {
            payment: updatedPayment,
        },
    });
}));
// Delete payment (admin only)
exports.deletePayment = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const payment = yield models_1.default.Payment.findByPk(id);
    if (!payment) {
        return next(new errorHandler_2.AppError('No payment found with that ID', 404));
    }
    yield payment.destroy();
    res.status(204).json({
        status: 'success',
        data: null,
    });
}));
/**
 * @desc Get payment status by payment reference ID
 * @route GET /api/v1/payments/status/:paymentRef
 * @access Public
 */
exports.getPaymentStatusByRef = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { paymentRef } = req.params;
    // First, verify the payment status with Konnect to get the latest update
    try {
        yield (0, konnectPayment_service_1.verifyPayment)(paymentRef);
    }
    catch (error) {
        // Log the error but don't block. The webhook might still come through,
        // or the payment was never initiated correctly.
        logger_1.default.warn(`Verification check for payment ref ${paymentRef} failed. It might not exist or there's an issue with Konnect.`, { error });
    }
    const payment = yield models_1.default.Payment.findOne({ where: { transactionId: paymentRef } });
    if (!payment) {
        // It's possible the webhook is just delayed, so 404 is appropriate.
        return next(new errorHandler_2.AppError('No payment found with that reference ID', 404));
    }
    res.status(200).json({
        status: 'success',
        data: {
            payment,
        },
    });
}));
