"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.exportAllRoomsCalendar = exports.exportRoomCalendar = exports.extractGuestDetailsFromEventSummary = exports.extractGuestDetails = exports.createRoomMappingFromRoom = exports.getRoomMappingsForRoom = exports.deleteRoomMapping = exports.updateRoomMapping = exports.createRoomMapping = exports.getRoomMappings = exports.previewMultiRoomCalendar = exports.importIcsFile = exports.previewIcsFile = exports.importReservations = exports.getReservationGuestInfo = exports.parseReservations = exports.deleteMultiRoomCalendar = exports.updateMultiRoomCalendar = exports.createMultiRoomCalendar = exports.getMultiRoomCalendar = exports.getMultiRoomCalendars = void 0;
const sequelize_1 = require("sequelize");
const errorHandler_1 = require("../utils/errorHandler");
const models_1 = __importDefault(require("../models"));
const multiRoomIcalSync_1 = require("../utils/multiRoomIcalSync");
const guestDetailExtraction_1 = require("../utils/guestDetailExtraction");
// Get all multi-room calendars
exports.getMultiRoomCalendars = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const calendars = yield models_1.default.MultiRoomCalendar.findAll({
        include: [
            {
                model: models_1.default.ExternalCalendar,
                as: 'roomMappings',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                        include: [
                            {
                                model: models_1.default.RoomType,
                                as: 'roomType',
                            },
                        ],
                    }
                ]
            }
        ],
        order: [['createdAt', 'DESC']],
    });
    // Add mapped rooms count and last sync status
    const calendarsWithStats = yield Promise.all(calendars.map((calendar) => __awaiter(void 0, void 0, void 0, function* () {
        var _a;
        const mappingsCount = ((_a = calendar.roomMappings) === null || _a === void 0 ? void 0 : _a.length) || 0;
        return Object.assign(Object.assign({}, calendar.toJSON()), { mappingsCount, syncStatus: calendar.lastSynced ? 'synced' : 'never' });
    })));
    res.status(200).json({
        status: 'success',
        results: calendarsWithStats.length,
        data: {
            calendars: calendarsWithStats,
        },
    });
}));
// Get calendar by ID
exports.getMultiRoomCalendar = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const calendar = yield models_1.default.MultiRoomCalendar.findByPk(id, {
        include: [
            {
                model: models_1.default.ExternalCalendar,
                as: 'roomMappings',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                        include: [
                            {
                                model: models_1.default.RoomType,
                                as: 'roomType',
                            },
                        ],
                    }
                ]
            }
        ]
    });
    if (!calendar) {
        return next(new errorHandler_1.AppError('Calendar not found', 404));
    }
    res.status(200).json({
        status: 'success',
        data: {
            calendar,
        },
    });
}));
// Create a new multi-room calendar
exports.createMultiRoomCalendar = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { name, source, calendarUrl, syncSettings } = req.body;
    if (!name || !source || !calendarUrl) {
        return next(new errorHandler_1.AppError('Name, source, and calendar URL are required', 400));
    }
    // Test calendar URL by trying to parse it
    try {
        const testEvents = yield (0, multiRoomIcalSync_1.parseMultiRoomCalendar)(calendarUrl, 0);
        console.log(`Calendar test successful: found ${testEvents.length} events`);
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Invalid calendar URL: ${error.message}`, 400));
    }
    const calendar = yield models_1.default.MultiRoomCalendar.create({
        name,
        source,
        calendarUrl,
        syncSettings: syncSettings || {},
        isActive: true,
    });
    res.status(201).json({
        status: 'success',
        data: {
            calendar,
        },
    });
}));
// Update calendar
exports.updateMultiRoomCalendar = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const { name, source, calendarUrl, syncSettings, isActive } = req.body;
    const calendar = yield models_1.default.MultiRoomCalendar.findByPk(id);
    if (!calendar) {
        return next(new errorHandler_1.AppError('Calendar not found', 404));
    }
    // Test calendar URL if it's being updated
    if (calendarUrl && calendarUrl !== calendar.calendarUrl) {
        try {
            const testEvents = yield (0, multiRoomIcalSync_1.parseMultiRoomCalendar)(calendarUrl, 0);
            console.log(`Calendar test successful: found ${testEvents.length} events`);
        }
        catch (error) {
            return next(new errorHandler_1.AppError(`Invalid calendar URL: ${error.message}`, 400));
        }
    }
    yield calendar.update({
        name: name || calendar.name,
        source: source || calendar.source,
        calendarUrl: calendarUrl || calendar.calendarUrl,
        syncSettings: syncSettings || calendar.syncSettings,
        isActive: isActive !== undefined ? isActive : calendar.isActive,
    });
    res.status(200).json({
        status: 'success',
        data: {
            calendar,
        },
    });
}));
// Delete calendar
exports.deleteMultiRoomCalendar = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const calendar = yield models_1.default.MultiRoomCalendar.findByPk(id);
    if (!calendar) {
        return next(new errorHandler_1.AppError('Calendar not found', 404));
    }
    yield calendar.destroy();
    res.status(204).json({
        status: 'success',
        data: null,
    });
}));
// Parse and preview reservations from calendar
exports.parseReservations = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const calendar = yield models_1.default.MultiRoomCalendar.findByPk(id);
    if (!calendar) {
        return next(new errorHandler_1.AppError('Calendar not found', 404));
    }
    try {
        const parsedData = yield (0, multiRoomIcalSync_1.parseAndMapReservations)(parseInt(id));
        // Update last sync attempt
        yield calendar.update({
            lastSynced: new Date()
        });
        res.status(200).json({
            status: 'success',
            data: {
                calendar: {
                    id: calendar.id,
                    name: calendar.name,
                    source: calendar.source
                },
                summary: {
                    totalEvents: parsedData.events.length,
                    reservations: parsedData.reservations.length,
                    conflicts: parsedData.conflicts.length,
                    unmappedEvents: parsedData.unmappedEvents.length
                },
                reservations: parsedData.reservations,
                conflicts: parsedData.conflicts,
                unmappedEvents: parsedData.unmappedEvents
            },
        });
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Failed to parse calendar: ${error.message}`, 500));
    }
}));
// Get guest info for specific reservation UID from ICS
exports.getReservationGuestInfo = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const { uid } = req.params;
    try {
        // Parse the calendar to get the specific reservation
        const parsedData = yield (0, multiRoomIcalSync_1.parseAndMapReservations)(parseInt(id));
        const allReservations = [...parsedData.reservations, ...parsedData.conflicts, ...parsedData.unmappedEvents];
        const reservation = allReservations.find(r => r.uid === uid);
        if (!reservation) {
            return next(new errorHandler_1.AppError('Reservation not found in calendar', 404));
        }
        // Extract available guest info from the reservation
        const guestInfo = {
            uid: reservation.uid,
            summary: reservation.summary,
            guestName: reservation.guestName,
            guestEmail: reservation.guestEmail,
            phoneNumber: reservation.phoneNumber,
            description: reservation.description,
            start: reservation.start,
            end: reservation.end,
            source: reservation.source
        };
        res.status(200).json({
            status: 'success',
            data: {
                guestInfo
            }
        });
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Failed to get guest info: ${error.message}`, 500));
    }
}));
// Import selected reservations with custom guest details and room mappings
exports.importReservations = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const { reservations, forceImport = false } = req.body;
    // Support both old format (reservationUids) and new format (reservations with details)
    if (req.body.reservationUids && Array.isArray(req.body.reservationUids)) {
        // Old format - just UIDs, use existing logic
        const results = yield (0, multiRoomIcalSync_1.importSelectedReservations)(parseInt(id), req.body.reservationUids, forceImport);
        return res.status(200).json({
            status: 'success',
            message: `Import completed: ${results.imported.length} imported, ${results.failed.length} failed, ${results.conflicts.length} conflicts`,
            data: results,
        });
    }
    // New format - reservations with custom details
    if (!reservations || !Array.isArray(reservations) || reservations.length === 0) {
        return next(new errorHandler_1.AppError('Reservations data is required', 400));
    }
    try {
        const results = yield (0, multiRoomIcalSync_1.importReservationsWithCustomDetails)(parseInt(id), reservations, forceImport);
        res.status(200).json({
            status: 'success',
            message: `Import completed: ${results.imported.length} imported, ${results.failed.length} failed, ${results.conflicts.length} conflicts`,
            data: results,
        });
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Import failed: ${error.message}`, 500));
    }
}));
// Preview events from an uploaded ICS file
exports.previewIcsFile = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { icsContent, roomMappings } = req.body;
    if (!icsContent) {
        return next(new errorHandler_1.AppError('ICS content is required', 400));
    }
    try {
        // Use the same logic as parseAndMapReservations but with file content
        const allReservations = yield (0, multiRoomIcalSync_1.parseIcsContent)(icsContent, roomMappings || []);
        res.status(200).json({
            status: 'success',
            data: {
                reservations: allReservations.reservations,
                conflicts: allReservations.conflicts,
                unmappedEvents: allReservations.unmappedEvents
            },
        });
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Failed to preview ICS file: ${error.message}`, 500));
    }
}));
// Import reservations from an uploaded ICS file
exports.importIcsFile = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { icsContent, reservations, roomMappings, forceImport } = req.body;
    if (!icsContent) {
        return next(new errorHandler_1.AppError('ICS content is required', 400));
    }
    if (!reservations || !Array.isArray(reservations)) {
        return next(new errorHandler_1.AppError('Reservations array is required', 400));
    }
    try {
        // Use the same import logic as importReservationsWithCustomDetails
        const results = yield (0, multiRoomIcalSync_1.importReservationsFromIcsContent)(icsContent, reservations, roomMappings || [], forceImport);
        res.status(200).json({
            status: 'success',
            message: `Import completed: ${results.imported.length} imported, ${results.failed.length} failed, ${results.conflicts.length} conflicts`,
            data: results,
        });
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Import failed: ${error.message}`, 500));
    }
}));
// Preview events from a multi-room calendar
exports.previewMultiRoomCalendar = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { calendarUrl } = req.body;
    if (!calendarUrl) {
        return next(new errorHandler_1.AppError('Calendar URL is required', 400));
    }
    try {
        const events = yield (0, multiRoomIcalSync_1.parseMultiRoomCalendar)(calendarUrl, 0);
        // Separate reservations from blocked time
        const reservations = events.filter(event => event.isReservation);
        const blockedTime = events.filter(event => !event.isReservation);
        res.status(200).json({
            status: 'success',
            data: {
                events: {
                    reservations: reservations.map(event => (Object.assign(Object.assign({}, event), { type: 'reservation' }))),
                    blockedTime: blockedTime.map(event => (Object.assign(Object.assign({}, event), { type: 'blocked' })))
                }
            },
        });
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Preview failed: ${error.message}`, 400));
    }
}));
// Get room mappings for a calendar
exports.getRoomMappings = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const mappings = yield models_1.default.ExternalCalendar.findAll({
        where: { multiRoomCalendarId: id },
        include: [
            {
                model: models_1.default.Room,
                as: 'room',
                include: [
                    {
                        model: models_1.default.RoomType,
                        as: 'roomType',
                    },
                ],
            }
        ],
        order: [['createdAt', 'DESC']],
    });
    res.status(200).json({
        status: 'success',
        results: mappings.length,
        data: {
            mappings,
        },
    });
}));
// Create room mapping
exports.createRoomMapping = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params; // calendar ID
    const { roomId, externalRoomIdentifier, mappingRules } = req.body;
    if (!roomId) {
        return next(new errorHandler_1.AppError('Room ID is required', 400));
    }
    // Check for duplicate mappings based on the new constraints
    if (externalRoomIdentifier) {
        const existingExternalIdMapping = yield models_1.default.ExternalCalendar.findOne({
            where: {
                multiRoomCalendarId: id,
                externalRoomIdentifier: externalRoomIdentifier
            }
        });
        if (existingExternalIdMapping) {
            return next(new errorHandler_1.AppError(`The external ID "${externalRoomIdentifier}" is already mapped to another room in this calendar.`, 409));
        }
    }
    const existingRoomMapping = yield models_1.default.ExternalCalendar.findOne({
        where: {
            multiRoomCalendarId: id,
            roomId: roomId
        }
    });
    if (existingRoomMapping) {
        return next(new errorHandler_1.AppError('This room is already mapped to this calendar. You can edit the existing mapping.', 409));
    }
    const mapping = yield models_1.default.ExternalCalendar.create({
        multiRoomCalendarId: id,
        roomId,
        name: `Room Mapping - ${roomId}`,
        externalRoomIdentifier: externalRoomIdentifier || null,
        mappingRules: mappingRules || [],
        isActive: true,
    });
    // Fetch the created mapping with includes
    const createdMapping = yield models_1.default.ExternalCalendar.findByPk(mapping.id, {
        include: [
            {
                model: models_1.default.Room,
                as: 'room',
                include: [
                    {
                        model: models_1.default.RoomType,
                        as: 'roomType',
                    },
                ],
            }
        ]
    });
    res.status(201).json({
        status: 'success',
        data: {
            mapping: createdMapping,
        },
    });
}));
// ... (existing controller code) ...
// Update room mapping
exports.updateRoomMapping = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { calendarId, mappingId } = req.params;
    const { externalRoomIdentifier, mappingRules, isActive } = req.body;
    const mapping = yield models_1.default.ExternalCalendar.findOne({
        where: {
            id: mappingId,
            multiRoomCalendarId: calendarId
        }
    });
    if (!mapping) {
        return next(new errorHandler_1.AppError('Mapping not found', 404));
    }
    // If the externalRoomIdentifier is being changed, check for duplicates
    if (externalRoomIdentifier && externalRoomIdentifier !== mapping.externalRoomIdentifier) {
        const existingMapping = yield models_1.default.ExternalCalendar.findOne({
            where: {
                multiRoomCalendarId: calendarId,
                externalRoomIdentifier: externalRoomIdentifier,
                id: { [sequelize_1.Op.ne]: mappingId } // Exclude the current mapping from the check
            }
        });
        if (existingMapping) {
            return next(new errorHandler_1.AppError(`The external ID "${externalRoomIdentifier}" is already in use by another room in this calendar.`, 409));
        }
    }
    yield mapping.update({
        externalRoomIdentifier: externalRoomIdentifier !== undefined ? externalRoomIdentifier : mapping.externalRoomIdentifier,
        mappingRules: mappingRules || mapping.mappingRules,
        isActive: isActive !== undefined ? isActive : mapping.isActive,
    });
    // Fetch updated mapping with includes
    const updatedMapping = yield models_1.default.ExternalCalendar.findByPk(mapping.id, {
        include: [
            {
                model: models_1.default.Room,
                as: 'room',
                include: [
                    {
                        model: models_1.default.RoomType,
                        as: 'roomType',
                    },
                ],
            }
        ]
    });
    res.status(200).json({
        status: 'success',
        data: {
            mapping: updatedMapping,
        },
    });
}));
// Delete room mapping
exports.deleteRoomMapping = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { calendarId, mappingId } = req.params;
    const mapping = yield models_1.default.ExternalCalendar.findOne({
        where: {
            id: mappingId,
            multiRoomCalendarId: calendarId
        }
    });
    if (!mapping) {
        return next(new errorHandler_1.AppError('Mapping not found', 404));
    }
    yield mapping.destroy();
    res.status(204).json({
        status: 'success',
        data: null,
    });
}));
// Get mappings for a specific room (used in room page)
exports.getRoomMappingsForRoom = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { roomId } = req.params;
    const mappings = yield models_1.default.ExternalCalendar.findAll({
        where: { roomId },
        include: [
            {
                model: models_1.default.MultiRoomCalendar,
                as: 'multiRoomCalendar'
            }
        ],
        order: [['createdAt', 'DESC']],
    });
    res.status(200).json({
        status: 'success',
        results: mappings.length,
        data: {
            mappings,
        },
    });
}));
// Create room mapping from room page
exports.createRoomMappingFromRoom = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { roomId } = req.params;
    const { calendarId, externalRoomIdentifier, mappingRules } = req.body;
    if (!calendarId) {
        return next(new errorHandler_1.AppError('Calendar ID is required', 400));
    }
    // Check if mapping already exists
    const existingMapping = yield models_1.default.ExternalCalendar.findOne({
        where: {
            multiRoomCalendarId: calendarId,
            roomId: roomId
        }
    });
    if (existingMapping) {
        return next(new errorHandler_1.AppError('Mapping for this calendar already exists', 400));
    }
    const mapping = yield models_1.default.ExternalCalendar.create({
        multiRoomCalendarId: calendarId,
        roomId,
        name: `Room Mapping - ${roomId}`,
        externalRoomIdentifier: externalRoomIdentifier || null,
        mappingRules: mappingRules || [],
        isActive: true,
    });
    // Fetch the created mapping with includes
    const createdMapping = yield models_1.default.ExternalCalendar.findByPk(mapping.id, {
        include: [
            {
                model: models_1.default.MultiRoomCalendar,
                as: 'multiRoomCalendar'
            }
        ]
    });
    res.status(201).json({
        status: 'success',
        data: {
            mapping: createdMapping,
        },
    });
}));
// Extract guest details from event summaries
exports.extractGuestDetails = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { events } = req.body;
    if (!events || !Array.isArray(events)) {
        return next(new errorHandler_1.AppError('Events array is required', 400));
    }
    try {
        const eventsWithGuestDetails = events.map(event => {
            const extractedDetails = (0, guestDetailExtraction_1.extractGuestDetailsFromSummary)(event.summary || event.title || '');
            const validation = (0, guestDetailExtraction_1.validateGuestDetails)(extractedDetails);
            return Object.assign(Object.assign({}, event), { extractedGuestDetails: extractedDetails, guestDetailsValidation: validation });
        });
        res.status(200).json({
            status: 'success',
            data: {
                events: eventsWithGuestDetails
            }
        });
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Failed to extract guest details: ${error.message}`, 500));
    }
}));
// Extract guest details from a single event summary
exports.extractGuestDetailsFromEventSummary = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { summary } = req.body;
    if (!summary) {
        return next(new errorHandler_1.AppError('Event summary is required', 400));
    }
    try {
        const extractedDetails = (0, guestDetailExtraction_1.extractGuestDetailsFromSummary)(summary);
        const validation = (0, guestDetailExtraction_1.validateGuestDetails)(extractedDetails);
        res.status(200).json({
            status: 'success',
            data: {
                extractedGuestDetails: extractedDetails,
                validation
            }
        });
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Failed to extract guest details: ${error.message}`, 500));
    }
}));
// Export all reservations for a specific room as an ICS file
exports.exportRoomCalendar = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { roomId } = req.params;
    try {
        const icsContent = yield (0, multiRoomIcalSync_1.generateIcs)(parseInt(roomId));
        res.setHeader('Content-Type', 'text/calendar');
        res.setHeader('Content-Disposition', `attachment; filename="room_${roomId}_calendar.ics"`);
        res.send(icsContent);
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Failed to generate ICS file: ${error.message}`, 500));
    }
}));
// Export all reservations for all rooms as a single ICS file
exports.exportAllRoomsCalendar = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const icsContent = yield (0, multiRoomIcalSync_1.generateAllRoomsIcs)();
        res.setHeader('Content-Type', 'text/calendar');
        res.setHeader('Content-Disposition', `attachment; filename="all_rooms_calendar.ics"`);
        res.send(icsContent);
    }
    catch (error) {
        return next(new errorHandler_1.AppError(`Failed to generate ICS file: ${error.message}`, 500));
    }
}));
