"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateFeedbackToken = exports.updateFeedback = exports.getPublishedFeedback = exports.getAllFeedback = exports.submitFeedback = exports.getFeedbackForm = void 0;
const sequelize_1 = require("sequelize");
const uuid_1 = require("uuid");
const errorHandler_1 = require("../utils/errorHandler");
const models_1 = __importDefault(require("../models"));
// Get feedback form by token (public route)
exports.getFeedbackForm = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { token } = req.params;
    if (!token) {
        return next(new errorHandler_1.AppError('Feedback token is required', 400));
    }
    // Validate token using FeedbackToken table (not expired, not used)
    const tokenRecord = yield models_1.default.FeedbackToken.findOne({
        where: {
            token,
            isUsed: false,
            expiresAt: { [sequelize_1.Op.gt]: new Date() },
        },
        include: [
            {
                model: models_1.default.Booking,
                as: 'booking',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                        include: [
                            { model: models_1.default.RoomType, as: 'roomType' },
                        ],
                    },
                    { model: models_1.default.Guest, as: 'guest' },
                ],
            },
        ],
    });
    if (!tokenRecord || !tokenRecord.booking) {
        return next(new errorHandler_1.AppError('Invalid feedback token or booking not found', 404));
    }
    // Check if feedback already exists for this token
    const existingFeedback = yield models_1.default.Feedback.findOne({
        where: { feedbackToken: token },
    });
    if (existingFeedback) {
        return res.status(200).json({
            status: 'success',
            data: {
                feedback: existingFeedback,
                alreadySubmitted: true,
            },
        });
    }
    res.status(200).json({
        status: 'success',
        data: {
            booking: tokenRecord.booking,
            token: tokenRecord.token,
            alreadySubmitted: false,
        },
    });
}));
// Submit feedback (public route)
exports.submitFeedback = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { token } = req.params;
    const { bookingId, overallRating, cleanlinessRating, serviceRating, locationRating, valueRating, comment, wouldRecommend, } = req.body;
    if (!token) {
        return next(new errorHandler_1.AppError('Feedback token is required', 400));
    }
    if (!bookingId || !overallRating) {
        return next(new errorHandler_1.AppError('Booking ID and overall rating are required', 400));
    }
    // Validate token: must exist, not used, not expired, and match booking
    const tokenRecord = yield models_1.default.FeedbackToken.findOne({
        where: {
            token,
            isUsed: false,
            expiresAt: { [sequelize_1.Op.gt]: new Date() },
        },
    });
    if (!tokenRecord) {
        return next(new errorHandler_1.AppError('Invalid or expired feedback token', 404));
    }
    if (tokenRecord.bookingId !== bookingId) {
        return next(new errorHandler_1.AppError('Token does not match booking', 400));
    }
    // Check if feedback already exists for this token
    const existingFeedback = yield models_1.default.Feedback.findOne({ where: { feedbackToken: token } });
    if (existingFeedback) {
        return next(new errorHandler_1.AppError('Feedback has already been submitted for this booking', 400));
    }
    // Verify booking exists and is completed
    const booking = yield models_1.default.Booking.findByPk(bookingId, {
        include: [
            {
                model: models_1.default.Room,
                as: 'room',
            },
            {
                model: models_1.default.Guest,
                as: 'guest',
            },
        ],
    });
    if (!booking) {
        return next(new errorHandler_1.AppError('Booking not found', 404));
    }
    if (booking.status !== 'completed') {
        return next(new errorHandler_1.AppError('Feedback can only be submitted for completed bookings', 400));
    }
    // Create feedback
    const feedback = yield models_1.default.Feedback.create({
        bookingId,
        guestId: booking.guestId,
        roomId: booking.roomId,
        overallRating,
        cleanlinessRating: cleanlinessRating || null,
        serviceRating: serviceRating || null,
        locationRating: locationRating || null,
        valueRating: valueRating || null,
        comment: comment || null,
        wouldRecommend: wouldRecommend !== undefined ? wouldRecommend : true,
        isPublished: true,
        feedbackToken: token,
        submittedAt: new Date(),
    });
    // Mark token as used
    yield tokenRecord.update({ isUsed: true, usedAt: new Date() });
    // Include related models in response
    const feedbackWithDetails = yield models_1.default.Feedback.findByPk(feedback.id, {
        include: [
            {
                model: models_1.default.Booking,
                as: 'booking',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                        include: [
                            {
                                model: models_1.default.RoomType,
                                as: 'roomType',
                            },
                        ],
                    },
                    {
                        model: models_1.default.Guest,
                        as: 'guest',
                    },
                ],
            },
        ],
    });
    res.status(201).json({
        status: 'success',
        data: {
            feedback: feedbackWithDetails,
        },
    });
}));
// Get all feedback (admin route)
exports.getAllFeedback = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { page = 1, limit = 10, isPublished, minRating } = req.query;
    const whereClause = {};
    if (isPublished !== undefined) {
        whereClause.isPublished = isPublished === 'true';
    }
    if (minRating) {
        whereClause.overallRating = { [sequelize_1.Op.gte]: parseInt(minRating) };
    }
    const offset = (parseInt(page) - 1) * parseInt(limit);
    const feedback = yield models_1.default.Feedback.findAndCountAll({
        where: whereClause,
        limit: parseInt(limit),
        offset,
        order: [['submittedAt', 'DESC']],
        include: [
            {
                model: models_1.default.Booking,
                as: 'booking',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                        include: [
                            {
                                model: models_1.default.RoomType,
                                as: 'roomType',
                            },
                        ],
                    },
                    {
                        model: models_1.default.Guest,
                        as: 'guest',
                    },
                ],
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        data: {
            feedback: feedback.rows,
            total: feedback.count,
            page: parseInt(page),
            limit: parseInt(limit),
            totalPages: Math.ceil(feedback.count / parseInt(limit)),
        },
    });
}));
// Get published feedback for public display
exports.getPublishedFeedback = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { limit = 6, minRating = 4 } = req.query;
    const feedback = yield models_1.default.Feedback.findAll({
        where: {
            isPublished: true,
            overallRating: { [sequelize_1.Op.gte]: parseInt(minRating) },
            comment: { [sequelize_1.Op.ne]: null }, // Only include feedback with comments
        },
        limit: parseInt(limit),
        order: [['submittedAt', 'DESC']],
        include: [
            {
                model: models_1.default.Booking,
                as: 'booking',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                        include: [
                            {
                                model: models_1.default.RoomType,
                                as: 'roomType',
                            },
                        ],
                    },
                    {
                        model: models_1.default.Guest,
                        as: 'guest',
                        attributes: ['firstName', 'lastName'], // Only include name for privacy
                    },
                ],
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        data: {
            feedback,
        },
    });
}));
// Update feedback publication status (admin route)
exports.updateFeedback = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { id } = req.params;
    const { isPublished } = req.body;
    const feedback = yield models_1.default.Feedback.findByPk(id);
    if (!feedback) {
        return next(new errorHandler_1.AppError('Feedback not found', 404));
    }
    yield feedback.update({
        isPublished: isPublished !== undefined ? isPublished : feedback.isPublished,
    });
    const updatedFeedback = yield models_1.default.Feedback.findByPk(id, {
        include: [
            {
                model: models_1.default.Booking,
                as: 'booking',
                include: [
                    {
                        model: models_1.default.Room,
                        as: 'room',
                        include: [
                            {
                                model: models_1.default.RoomType,
                                as: 'roomType',
                            },
                        ],
                    },
                    {
                        model: models_1.default.Guest,
                        as: 'guest',
                    },
                ],
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        data: {
            feedback: updatedFeedback,
        },
    });
}));
// Generate feedback token for a booking (used internally)
const generateFeedbackToken = () => {
    return (0, uuid_1.v4)();
};
exports.generateFeedbackToken = generateFeedbackToken;
