"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDashboardStats = void 0;
const sequelize_1 = require("sequelize");
const models_1 = __importDefault(require("../models"));
const Booking = models_1.default.Booking;
const Guest = models_1.default.Guest;
const Room = models_1.default.Room;
// Helper function to check if two date ranges overlap
const hasOverlap = (start1, end1, start2, end2) => {
    return start1 < end2 && end1 > start2;
};
// Helper function to calculate nights within a specific period
const calculateNightsInPeriod = (bookingStart, bookingEnd, periodStart, periodEnd) => {
    const actualStart = new Date(Math.max(bookingStart.getTime(), periodStart.getTime()));
    const actualEnd = new Date(Math.min(bookingEnd.getTime(), periodEnd.getTime()));
    if (actualStart >= actualEnd)
        return 0;
    return Math.ceil((actualEnd.getTime() - actualStart.getTime()) / (1000 * 3600 * 24));
};
// Get dashboard statistics
const getDashboardStats = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        // Get date range from query parameters
        const { startDate, endDate } = req.query;
        let periodStart;
        let periodEnd;
        // If startDate and endDate are provided, use them
        if (startDate && endDate) {
            periodStart = new Date(startDate);
            periodStart.setHours(0, 0, 0, 0);
            periodEnd = new Date(endDate);
            periodEnd.setHours(23, 59, 59, 999);
        }
        else {
            // Default to current month if no date range provided
            const now = new Date();
            periodStart = new Date(now.getFullYear(), now.getMonth(), 1);
            periodEnd = new Date(now.getFullYear(), now.getMonth() + 1, 0, 23, 59, 59, 999);
        }
        // Get current date for today's stats
        const now = new Date();
        const todayStart = new Date(now);
        todayStart.setHours(0, 0, 0, 0);
        const todayEnd = new Date(now);
        todayEnd.setHours(23, 59, 59, 999);
        // FIXED: Count bookings that are ACTIVE during the period (not just check-ins)
        const periodBookings = yield Booking.count({
            where: {
                [sequelize_1.Op.and]: [
                    {
                        checkInDate: {
                            [sequelize_1.Op.lte]: periodEnd
                        }
                    },
                    {
                        checkOutDate: {
                            [sequelize_1.Op.gte]: periodStart
                        }
                    },
                    {
                        status: {
                            [sequelize_1.Op.ne]: 'cancelled'
                        }
                    }
                ]
            }
        });
        // Today's bookings (check-ins only for today's activity)
        const todayBookings = yield Booking.count({
            where: {
                checkInDate: {
                    [sequelize_1.Op.between]: [todayStart, todayEnd]
                },
                status: {
                    [sequelize_1.Op.ne]: 'cancelled'
                }
            }
        });
        // Total guests
        const totalGuests = yield Guest.count();
        // Get all rooms for availability calculation
        const allRooms = yield Room.findAll();
        const totalRooms = allRooms.length;
        // Get bookings that overlap with the specified period for occupancy calculation
        const overlappingBookings = yield Booking.findAll({
            where: {
                [sequelize_1.Op.and]: [
                    {
                        checkInDate: {
                            [sequelize_1.Op.lte]: periodEnd
                        }
                    },
                    {
                        checkOutDate: {
                            [sequelize_1.Op.gte]: periodStart
                        }
                    },
                    {
                        status: {
                            [sequelize_1.Op.ne]: 'cancelled'
                        }
                    }
                ]
            },
            include: [
                { model: Room, as: 'room' }
            ]
        });
        // FIXED: Calculate room availability with proper overlap detection
        const isRoomAvailableForRange = (roomId) => {
            return !overlappingBookings.some(booking => {
                if (booking.get('roomId') !== roomId)
                    return false;
                const resStart = new Date(booking.get('checkInDate'));
                const resEnd = new Date(booking.get('checkOutDate'));
                // FIXED: Proper overlap detection
                return hasOverlap(resStart, resEnd, periodStart, periodEnd);
            });
        };
        const availableRooms = allRooms.filter(room => isRoomAvailableForRange(room.get('id')));
        const occupiedRooms = totalRooms - availableRooms.length;
        const occupancyRate = totalRooms > 0 ? (occupiedRooms / totalRooms) * 100 : 0;
        // FIXED: Revenue calculation for only nights within the specified period
        let periodRevenue = 0;
        overlappingBookings.forEach(booking => {
            const checkInDate = new Date(booking.get('checkInDate'));
            const checkOutDate = new Date(booking.get('checkOutDate'));
            // Calculate only nights that fall within the period
            const nightsInPeriod = calculateNightsInPeriod(checkInDate, checkOutDate, periodStart, periodEnd);
            const room = booking.get('room');
            const pricePerNight = room ? parseFloat(room.pricePerNight) : 0;
            periodRevenue += pricePerNight * nightsInPeriod;
        });
        return res.status(200).json({
            status: 'success',
            data: {
                period: {
                    startDate: periodStart.toISOString().split('T')[0],
                    endDate: periodEnd.toISOString().split('T')[0]
                },
                bookings: {
                    today: todayBookings,
                    period: periodBookings
                },
                guests: {
                    total: totalGuests
                },
                rooms: {
                    total: totalRooms,
                    available: availableRooms.length,
                    occupied: occupiedRooms,
                    occupancyRate: occupancyRate.toFixed(2)
                },
                revenue: {
                    period: periodRevenue.toFixed(2)
                }
            }
        });
    }
    catch (error) {
        next(error);
    }
});
exports.getDashboardStats = getDashboardStats;
