"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteContactMessage = exports.replyToContactMessage = exports.updateContactMessage = exports.createContactMessage = exports.getContactMessageById = exports.getAllContactMessages = void 0;
const errorHandler_1 = require("../utils/errorHandler");
const models_1 = __importDefault(require("../models"));
const email_service_1 = __importDefault(require("../services/email.service"));
// Get all contact messages
exports.getAllContactMessages = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { status, assignedTo } = req.query;
    const whereClause = {};
    if (status)
        whereClause.status = status;
    if (assignedTo)
        whereClause.assignedTo = assignedTo;
    const contactMessages = yield models_1.default.ContactMessage.findAll({
        where: whereClause,
        include: [
            {
                model: models_1.default.User,
                as: 'assignedUser',
                attributes: ['id', 'firstName', 'lastName', 'email'],
            },
        ],
        order: [['createdAt', 'DESC']],
    });
    res.status(200).json({
        status: 'success',
        results: contactMessages.length,
        data: {
            contactMessages,
        },
    });
}));
// Get contact message by ID
exports.getContactMessageById = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const contactMessage = yield models_1.default.ContactMessage.findByPk(req.params.id, {
        include: [
            {
                model: models_1.default.User,
                as: 'assignedUser',
                attributes: ['id', 'firstName', 'lastName', 'email'],
            },
        ],
    });
    if (!contactMessage) {
        return next(new errorHandler_1.AppError('No contact message found with that ID', 404));
    }
    // If message is new, mark it as read
    if (contactMessage.status === 'new') {
        yield contactMessage.update({ status: 'read' });
    }
    res.status(200).json({
        status: 'success',
        data: {
            contactMessage,
        },
    });
}));
// Create a new contact message
exports.createContactMessage = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { name, email, phone, subject, message } = req.body;
    const contactMessage = yield models_1.default.ContactMessage.create({
        name,
        email,
        phone,
        subject,
        message,
        status: 'new',
    });
    // Send email notification to owner
    try {
        yield email_service_1.default.sendContactNotification(name, email, subject, message, phone);
    }
    catch (error) {
        console.error('Failed to send contact notification email:', error);
        // Don't fail the request if email fails
    }
    res.status(201).json({
        status: 'success',
        data: {
            contactMessage,
        },
    });
}));
// Update contact message
exports.updateContactMessage = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { status, assignedTo } = req.body;
    const contactMessage = yield models_1.default.ContactMessage.findByPk(req.params.id);
    if (!contactMessage) {
        return next(new errorHandler_1.AppError('No contact message found with that ID', 404));
    }
    // If assignedTo is provided, check if user exists
    if (assignedTo) {
        const user = yield models_1.default.User.findByPk(assignedTo);
        if (!user) {
            return next(new errorHandler_1.AppError('No user found with that ID', 404));
        }
    }
    // Update contact message
    yield contactMessage.update({
        status: status || contactMessage.status,
        assignedTo: assignedTo !== undefined ? assignedTo : contactMessage.assignedTo,
    });
    // Get updated contact message with relations
    const updatedContactMessage = yield models_1.default.ContactMessage.findByPk(req.params.id, {
        include: [
            {
                model: models_1.default.User,
                as: 'assignedUser',
                attributes: ['id', 'firstName', 'lastName', 'email'],
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        data: {
            contactMessage: updatedContactMessage,
        },
    });
}));
// Reply to contact message
exports.replyToContactMessage = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const { replyMessage } = req.body;
    if (!replyMessage) {
        return next(new errorHandler_1.AppError('Reply message is required', 400));
    }
    const contactMessage = yield models_1.default.ContactMessage.findByPk(req.params.id);
    if (!contactMessage) {
        return next(new errorHandler_1.AppError('No contact message found with that ID', 404));
    }
    // Update contact message with reply
    yield contactMessage.update({
        status: 'replied',
        replyMessage,
        repliedAt: new Date(),
        assignedTo: req.user.id,
    });
    // Send email reply to the guest
    try {
        yield email_service_1.default.sendContactReply(contactMessage.email, contactMessage.name, replyMessage, contactMessage.subject);
    }
    catch (error) {
        console.error('Failed to send reply email:', error);
        // Don't fail the request if email fails
    }
    // Get updated contact message with relations
    const updatedContactMessage = yield models_1.default.ContactMessage.findByPk(req.params.id, {
        include: [
            {
                model: models_1.default.User,
                as: 'assignedUser',
                attributes: ['id', 'firstName', 'lastName', 'email'],
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        data: {
            contactMessage: updatedContactMessage,
        },
    });
}));
// Delete contact message
exports.deleteContactMessage = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const contactMessage = yield models_1.default.ContactMessage.findByPk(req.params.id);
    if (!contactMessage) {
        return next(new errorHandler_1.AppError('No contact message found with that ID', 404));
    }
    yield contactMessage.destroy();
    res.status(204).json({
        status: 'success',
        data: null,
    });
}));
