"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateExternalCalendarSettings = exports.getExternalCalendarEvents = exports.syncExternalCalendar = exports.updateAvailability = exports.getRoomAvailability = exports.getAvailability = exports.previewIcsFile = exports.importFromExternalCalendar = exports.previewExternalCalendar = exports.getRoomCalendarFeed = exports.syncCalendars = exports.deleteExternalCalendar = exports.updateExternalCalendar = exports.addExternalCalendar = exports.getExternalCalendars = void 0;
const sequelize_1 = require("sequelize");
const models_1 = __importStar(require("../models"));
const icalSync_1 = require("../utils/icalSync");
const nodeIcal = __importStar(require("node-ical"));
// Helper function to check room availability for a date range (same as booking controller)
const isRoomAvailable = (roomId, checkInDate, checkOutDate, excludeBookingId) => __awaiter(void 0, void 0, void 0, function* () {
    const whereClause = {
        roomId,
        status: {
            [sequelize_1.Op.notIn]: ['cancelled'],
        },
        [sequelize_1.Op.or]: [
            {
                // Bookings that start during the requested period (excluding checkout date)
                checkInDate: {
                    [sequelize_1.Op.gte]: checkInDate,
                    [sequelize_1.Op.lt]: checkOutDate,
                },
            },
            {
                // Bookings that end during the requested period (excluding checkin date)
                checkOutDate: {
                    [sequelize_1.Op.gt]: checkInDate,
                    [sequelize_1.Op.lte]: checkOutDate,
                },
            },
            {
                // Bookings that span the entire requested period
                [sequelize_1.Op.and]: [
                    {
                        checkInDate: {
                            [sequelize_1.Op.lt]: checkInDate,
                        },
                    },
                    {
                        checkOutDate: {
                            [sequelize_1.Op.gt]: checkOutDate,
                        },
                    },
                ],
            },
        ],
    };
    // Exclude current booking for updates
    if (excludeBookingId) {
        whereClause.id = {
            [sequelize_1.Op.ne]: excludeBookingId,
        };
    }
    const conflictingBookings = yield models_1.default.Booking.count({
        where: whereClause,
    });
    return conflictingBookings === 0;
});
/**
 * Get all external calendars for a room
 * @route GET /api/rooms/:roomId/calendars
 */
const getExternalCalendars = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        // Get external calendars for this room
        const calendars = yield models_1.default.ExternalCalendar.findAll({
            where: { roomId },
            order: [['id', 'ASC']]
        });
        return res.status(200).json(calendars);
    }
    catch (error) {
        console.error('Error getting external calendars:', error);
        return res.status(500).json({ message: 'Failed to retrieve external calendars', error: error.message });
    }
});
exports.getExternalCalendars = getExternalCalendars;
/**
 * Add a new external calendar for a room
 * @route POST /api/rooms/:roomId/calendars
 */
const addExternalCalendar = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        const { name, source, calendarUrl } = req.body;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        // Check if calendar URL is valid by trying to fetch it
        try {
            // We'll try to sync external calendars after creating the new one
            // This just checks if we can create the object
            const calendar = yield models_1.default.ExternalCalendar.create({
                roomId,
                name,
                source,
                calendarUrl,
                isActive: true
            });
            // Now try to sync
            try {
                yield (0, icalSync_1.syncExternalCalendars)(parseInt(roomId));
                // Update last synced timestamp
                yield calendar.update({
                    lastSynced: new Date()
                });
            }
            catch (syncError) {
                console.error('Warning: Calendar created but sync failed:', syncError);
                // We don't fail the request if sync fails, just log a warning
            }
            return res.status(201).json(calendar);
        }
        catch (error) {
            return res.status(400).json({ message: 'Invalid calendar URL or unable to sync calendar', error: error.message });
        }
    }
    catch (error) {
        console.error('Error adding external calendar:', error);
        return res.status(500).json({ message: 'Failed to add external calendar', error: error.message });
    }
});
exports.addExternalCalendar = addExternalCalendar;
/**
 * Update an external calendar
 * @route PUT /api/rooms/:roomId/calendars/:calendarId
 */
const updateExternalCalendar = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { calendarId } = req.params;
        const { name, source, calendarUrl, isActive } = req.body;
        // Find the calendar
        const calendar = yield models_1.default.ExternalCalendar.findByPk(calendarId);
        if (!calendar) {
            return res.status(404).json({ message: 'External calendar not found' });
        }
        // Update the calendar
        yield calendar.update({
            name: name !== undefined ? name : calendar.name,
            source: source !== undefined ? source : calendar.source,
            calendarUrl: calendarUrl !== undefined ? calendarUrl : calendar.calendarUrl,
            isActive: isActive !== undefined ? isActive : calendar.isActive
        });
        // If calendar URL changed, verify it works by syncing
        if (calendarUrl && calendarUrl !== calendar.calendarUrl) {
            try {
                yield (0, icalSync_1.syncExternalCalendars)(calendar.roomId);
                // Update last synced timestamp
                yield calendar.update({
                    lastSynced: new Date()
                });
            }
            catch (error) {
                console.error('Warning: Calendar updated but sync failed:', error);
                // We don't fail the request if sync fails, just log a warning
            }
        }
        return res.status(200).json(calendar);
    }
    catch (error) {
        console.error('Error updating external calendar:', error);
        return res.status(500).json({ message: 'Failed to update external calendar', error: error.message });
    }
});
exports.updateExternalCalendar = updateExternalCalendar;
/**
 * Delete an external calendar
 * @route DELETE /api/rooms/:roomId/calendars/:calendarId
 */
const deleteExternalCalendar = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { calendarId } = req.params;
        // Find the calendar
        const calendar = yield models_1.default.ExternalCalendar.findByPk(calendarId);
        if (!calendar) {
            return res.status(404).json({ message: 'External calendar not found' });
        }
        // Delete the calendar
        yield calendar.destroy();
        return res.status(204).send();
    }
    catch (error) {
        console.error('Error deleting external calendar:', error);
        return res.status(500).json({ message: 'Failed to delete external calendar', error: error.message });
    }
});
exports.deleteExternalCalendar = deleteExternalCalendar;
/**
 * Sync external calendars for a room
 * @route POST /api/rooms/:roomId/calendars/sync
 */
const syncCalendars = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        yield (0, icalSync_1.syncExternalCalendars)(parseInt(roomId));
        // Update last synced timestamp for all calendars for this room
        yield models_1.default.ExternalCalendar.update({ lastSynced: new Date() }, { where: { roomId, isActive: true } });
        return res.status(200).json({ message: 'Calendars synced successfully' });
    }
    catch (error) {
        console.error('Error syncing calendars:', error);
        return res.status(500).json({ message: 'Failed to sync calendars', error: error.message });
    }
});
exports.syncCalendars = syncCalendars;
/**
 * Get iCal feed for a room
 * @route GET /api/rooms/:roomId/calendar.ics
 */
const getRoomCalendarFeed = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        const calendarData = yield (0, icalSync_1.generateRoomCalendarFeed)(parseInt(roomId));
        res.setHeader('Content-Type', 'text/calendar');
        res.setHeader('Content-Disposition', `attachment; filename="room-${roomId}.ics"`);
        return res.send(calendarData);
    }
    catch (error) {
        console.error('Error generating room calendar feed:', error);
        return res.status(500).json({ message: 'Failed to generate calendar feed', error: error.message });
    }
});
exports.getRoomCalendarFeed = getRoomCalendarFeed;
/**
 * Preview reservations from an external calendar to identify conflicts
 * @route GET /api/calendars/:calendarId/preview
 */
const previewExternalCalendar = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { calendarId } = req.params;
        const calendar = yield models_1.default.ExternalCalendar.findByPk(calendarId);
        if (!calendar || !calendar.isActive) {
            return res.status(404).json({ message: 'Active external calendar not found.' });
        }
        const externalEvents = yield (0, icalSync_1.fetchCalendar)(calendar.calendarUrl);
        const preview = [];
        for (const event of externalEvents) {
            const conflictingBooking = yield models_1.default.Booking.findOne({
                where: {
                    roomId: calendar.roomId,
                    status: { [sequelize_1.Op.ne]: 'cancelled' },
                    [sequelize_1.Op.or]: [
                        {
                            // Bookings that start during the requested period (excluding checkout date)
                            checkInDate: {
                                [sequelize_1.Op.gte]: event.start,
                                [sequelize_1.Op.lt]: event.end,
                            },
                        },
                        {
                            // Bookings that end during the requested period (excluding checkin date)
                            checkOutDate: {
                                [sequelize_1.Op.gt]: event.start,
                                [sequelize_1.Op.lt]: event.end,
                            },
                        },
                        {
                            // Bookings that span the entire requested period
                            [sequelize_1.Op.and]: [
                                {
                                    checkInDate: {
                                        [sequelize_1.Op.lt]: event.start,
                                    },
                                },
                                {
                                    checkOutDate: {
                                        [sequelize_1.Op.gt]: event.end,
                                    },
                                },
                            ],
                        },
                    ],
                }
            });
            preview.push({
                uid: event.uid,
                summary: event.summary,
                start: event.start,
                end: event.end,
                isConflict: !!conflictingBooking,
                conflictReason: conflictingBooking ? `Overlaps with existing booking ID: ${conflictingBooking.id}` : null
            });
        }
        return res.status(200).json(preview);
    }
    catch (error) {
        console.error('Error previewing external calendar:', error);
        return res.status(500).json({ message: 'Failed to preview calendar', error: error.message });
    }
});
exports.previewExternalCalendar = previewExternalCalendar;
/**
 * Import reservations from a previewed external calendar
 * @route POST /api/calendars/import
 */
const importFromExternalCalendar = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    const { roomId, eventsToImport } = req.body;
    console.log('Import request received:', { roomId, eventsCount: eventsToImport === null || eventsToImport === void 0 ? void 0 : eventsToImport.length });
    if (!roomId || !Array.isArray(eventsToImport) || eventsToImport.length === 0) {
        return res.status(400).json({ message: 'Room ID and a list of events to import are required.' });
    }
    // Check if room exists and is not in maintenance (same as booking validation)
    const room = yield models_1.default.Room.findByPk(roomId);
    if (!room) {
        return res.status(404).json({ message: 'Room not found.' });
    }
    if (room.status === 'maintenance') {
        return res.status(400).json({ message: 'Room is under maintenance and not available for booking.' });
    }
    const transaction = yield models_1.sequelize.transaction();
    let importedCount = 0;
    const errors = [];
    try {
        for (const event of eventsToImport) {
            try {
                console.log('Processing event:', event);
                // Parse and validate dates (same as booking validation)
                const checkIn = new Date(event.start);
                const checkOut = new Date(event.end);
                // Validate dates
                if (isNaN(checkIn.getTime()) || isNaN(checkOut.getTime())) {
                    errors.push({ uid: event.uid, error: 'Invalid date format' });
                    continue;
                }
                if (checkIn >= checkOut) {
                    errors.push({ uid: event.uid, error: 'Check-in date must be before check-out date' });
                    continue;
                }
                // Format dates properly for DATEONLY fields
                const checkInDate = checkIn.toISOString().split('T')[0]; // YYYY-MM-DD
                const checkOutDate = checkOut.toISOString().split('T')[0]; // YYYY-MM-DD
                console.log('Formatted dates:', { checkInDate, checkOutDate });
                // Check if room is available for the requested dates (same as booking validation)
                const isAvailable = yield isRoomAvailable(parseInt(roomId), checkIn, checkOut);
                if (!isAvailable) {
                    errors.push({ uid: event.uid, error: 'Room is not available for the selected dates' });
                    continue;
                }
                // Calculate number of nights and total price (same as booking validation)
                const oneDay = 24 * 60 * 60 * 1000;
                const nights = Math.round(Math.abs((checkOut.getTime() - checkIn.getTime()) / oneDay));
                const pricePerNight = room.pricePerNight;
                const discount = room.discount || 0;
                const totalPrice = nights * pricePerNight * (1 - (discount / 100));
                // Create a valid email by sanitizing the UID (remove @ symbols and special chars)
                const sanitizedUid = event.uid.replace(/[@.]/g, '-').replace(/[^a-zA-Z0-9-]/g, '').substring(0, 50);
                const placeholderEmail = `external-${sanitizedUid}@placeholder.com`;
                console.log('Event has guestDetails:', !!event.guestDetails);
                console.log('Event guestDetails:', event.guestDetails);
                console.log('Event has selectedGuestId:', !!event.selectedGuestId);
                console.log('Event selectedGuestId:', event.selectedGuestId);
                let guest;
                if (event.selectedGuestId) {
                    // Use existing guest
                    guest = yield models_1.default.Guest.findByPk(event.selectedGuestId, { transaction });
                    if (!guest) {
                        errors.push({ uid: event.uid, error: `Selected guest with ID ${event.selectedGuestId} not found` });
                        continue;
                    }
                    console.log('Using existing guest:', guest.firstName, guest.lastName);
                }
                else {
                    // Use provided guest details or create defaults
                    const guestData = event.guestDetails || {
                        firstName: ((_a = event.summary) === null || _a === void 0 ? void 0 : _a.split(' ')[0]) || 'External',
                        lastName: ((_b = event.summary) === null || _b === void 0 ? void 0 : _b.split(' ')[1]) || 'Guest',
                        email: placeholderEmail,
                        phone: '+1-000-000-0000',
                        address: 'External booking - no address provided',
                        city: 'Unknown',
                        country: 'Unknown'
                    };
                    console.log('Final guest data being used:', guestData);
                    // Ensure email is valid - if provided email is empty, use placeholder
                    if (!guestData.email || guestData.email.trim() === '') {
                        guestData.email = placeholderEmail;
                    }
                    // Create guest with all required fields
                    const [createdGuest] = yield models_1.default.Guest.findOrCreate({
                        where: { email: guestData.email },
                        defaults: guestData,
                        transaction
                    });
                    guest = createdGuest;
                }
                // Create booking with proper validation and pricing
                const booking = yield models_1.default.Booking.create({
                    roomId: parseInt(roomId),
                    guestId: guest.id,
                    checkInDate: checkInDate,
                    checkOutDate: checkOutDate,
                    status: 'confirmed',
                    specialRequests: `Imported from iCal: ${event.summary || 'No title'}`,
                    totalPrice: totalPrice // Use calculated price instead of 0
                }, { transaction });
                console.log('Booking created:', booking.id);
                // Create room availability records for each day of the stay (same as booking validation)
                importedCount++;
            }
            catch (error) {
                console.error('Error processing individual event:', error);
                errors.push({ uid: event.uid, error: error.message, details: error.stack });
            }
        }
        // Update room status to 'reserved' if any bookings were imported (same as booking validation)
        if (importedCount > 0) {
            yield room.update({
                status: 'reserved',
            }, { transaction });
        }
        yield transaction.commit();
        console.log('Transaction committed. Imported:', importedCount);
        if (errors.length > 0) {
            console.log('Errors occurred:', errors);
            return res.status(207).json({
                message: `Partially completed: Imported ${importedCount} of ${eventsToImport.length} reservations.`,
                importedCount,
                errors
            });
        }
        return res.status(201).json({
            message: `Successfully imported ${importedCount} reservations.`,
            importedCount
        });
    }
    catch (error) {
        yield transaction.rollback();
        console.error('Error importing from external calendar:', error);
        return res.status(500).json({
            message: 'Failed to import reservations',
            error: error.message,
            details: error.stack
        });
    }
});
exports.importFromExternalCalendar = importFromExternalCalendar;
/**
 * Preview reservations from an uploaded .ics file to identify conflicts
 * @route POST /api/calendars/preview-ics
 */
const previewIcsFile = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId, icsContent } = req.body;
        if (!roomId || !icsContent) {
            return res.status(400).json({ message: 'Room ID and iCal content are required.' });
        }
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found.' });
        }
        // Parse the iCal content
        const parsedCal = nodeIcal.parseICS(icsContent);
        const events = [];
        for (const k in parsedCal) {
            const event = parsedCal[k];
            if (event.type === 'VEVENT') {
                events.push({
                    uid: event.uid,
                    summary: event.summary,
                    start: event.start,
                    end: event.end,
                    source: 'uploaded_file'
                });
            }
        }
        const preview = [];
        for (const event of events) {
            const conflictingBooking = yield models_1.default.Booking.findOne({
                where: {
                    roomId,
                    status: { [sequelize_1.Op.ne]: 'cancelled' },
                    [sequelize_1.Op.or]: [
                        {
                            // Bookings that start during the requested period
                            checkInDate: {
                                [sequelize_1.Op.gte]: event.start,
                                [sequelize_1.Op.lte]: event.end,
                            },
                        },
                        {
                            // Bookings that end during the requested period
                            checkOutDate: {
                                [sequelize_1.Op.gte]: event.start,
                                [sequelize_1.Op.lte]: event.end,
                            },
                        },
                        {
                            // Bookings that span the entire requested period
                            [sequelize_1.Op.and]: [
                                {
                                    checkInDate: {
                                        [sequelize_1.Op.lt]: event.start,
                                    },
                                },
                                {
                                    checkOutDate: {
                                        [sequelize_1.Op.gt]: event.end,
                                    },
                                },
                            ],
                        },
                    ],
                }
            });
            preview.push({
                uid: event.uid,
                summary: event.summary,
                start: event.start,
                end: event.end,
                isConflict: !!conflictingBooking,
                conflictReason: conflictingBooking ? `Overlaps with existing booking ID: ${conflictingBooking.id}` : null
            });
        }
        return res.status(200).json(preview);
    }
    catch (error) {
        console.error('Error previewing ICS file:', error);
        return res.status(500).json({ message: 'Failed to preview ICS file', error: error.message });
    }
});
exports.previewIcsFile = previewIcsFile;
// Public routes
const getAvailability = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { startDate, endDate } = req.query;
        // Room availability is now booking-based only - this endpoint is deprecated
        return res.status(200).json({
            status: 'success',
            data: { availability: [] }
        });
    }
    catch (error) {
        console.error('Error getting availability:', error);
        return res.status(500).json({ message: 'Failed to retrieve availability', error: error.message });
    }
});
exports.getAvailability = getAvailability;
const getRoomAvailability = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        const { startDate, endDate } = req.query;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        // Parse dates
        const start = new Date(startDate);
        const end = new Date(endDate);
        // Generate array of all dates in the range
        const allDates = [];
        const currentDate = new Date(start);
        while (currentDate <= end) {
            allDates.push(currentDate.toISOString().split('T')[0]);
            currentDate.setDate(currentDate.getDate() + 1);
        }
        // Get unavailable dates using booking-based logic (same as checkRoomAvailability)
        const unavailableBookings = yield models_1.default.Booking.findAll({
            attributes: ['checkInDate', 'checkOutDate'],
            where: {
                roomId,
                status: {
                    [sequelize_1.Op.notIn]: ['cancelled'],
                },
                [sequelize_1.Op.or]: [
                    {
                        // Bookings that start during the requested period
                        checkInDate: {
                            [sequelize_1.Op.gte]: start,
                            [sequelize_1.Op.lte]: end,
                        },
                    },
                    {
                        // Bookings that end during the requested period
                        checkOutDate: {
                            [sequelize_1.Op.gte]: start,
                            [sequelize_1.Op.lte]: end,
                        },
                    },
                    {
                        // Bookings that span the entire requested period
                        [sequelize_1.Op.and]: [
                            {
                                checkInDate: {
                                    [sequelize_1.Op.lt]: start,
                                },
                            },
                            {
                                checkOutDate: {
                                    [sequelize_1.Op.gt]: end,
                                },
                            },
                        ],
                    },
                ],
            },
            raw: true,
        });
        // Calculate unavailable dates from bookings
        const unavailableDates = new Set();
        unavailableBookings.forEach((booking) => {
            const checkIn = new Date(booking.checkInDate);
            const checkOut = new Date(booking.checkOutDate);
            // Add all dates from check-in to check-out (exclusive of check-out)
            const bookingDate = new Date(checkIn);
            while (bookingDate < checkOut) {
                const dateStr = bookingDate.toISOString().split('T')[0];
                if (allDates.includes(dateStr)) {
                    unavailableDates.add(dateStr);
                }
                bookingDate.setDate(bookingDate.getDate() + 1);
            }
        });
        // Build availability response
        const availability = allDates.map(date => ({
            date,
            isAvailable: !unavailableDates.has(date),
            roomId: parseInt(roomId)
        }));
        return res.status(200).json({
            status: 'success',
            data: { availability }
        });
    }
    catch (error) {
        console.error('Error getting room availability:', error);
        return res.status(500).json({ message: 'Failed to retrieve room availability', error: error.message });
    }
});
exports.getRoomAvailability = getRoomAvailability;
// Protected routes
const updateAvailability = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId, date, isAvailable, price } = req.body;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        // Room availability is now booking-based only - manual blocks should be handled as special bookings
        return res.status(200).json({
            status: 'success',
            data: { availability: [] }
        });
    }
    catch (error) {
        console.error('Error updating availability:', error);
        return res.status(500).json({ message: 'Failed to update availability', error: error.message });
    }
});
exports.updateAvailability = updateAvailability;
const syncExternalCalendar = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.body;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        yield (0, icalSync_1.syncExternalCalendars)(parseInt(roomId));
        // Update last synced timestamp for all calendars for this room
        yield models_1.default.ExternalCalendar.update({ lastSynced: new Date() }, { where: { roomId, isActive: true } });
        return res.status(200).json({ message: 'Calendars synced successfully' });
    }
    catch (error) {
        console.error('Error syncing calendars:', error);
        return res.status(500).json({ message: 'Failed to sync calendars', error: error.message });
    }
});
exports.syncExternalCalendar = syncExternalCalendar;
const getExternalCalendarEvents = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        const { startDate, endDate } = req.query;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return res.status(404).json({ message: 'Room not found' });
        }
        const calendars = yield models_1.default.ExternalCalendar.findAll({
            where: { roomId, isActive: true },
            // Room availability is now booking-based only - no separate availability table
        });
        return res.status(200).json({
            status: 'success',
            data: { calendars }
        });
    }
    catch (error) {
        console.error('Error getting external calendar events:', error);
        return res.status(500).json({ message: 'Failed to retrieve calendar events', error: error.message });
    }
});
exports.getExternalCalendarEvents = getExternalCalendarEvents;
const updateExternalCalendarSettings = (req, res) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { calendarId } = req.params;
        const { name, source, calendarUrl, isActive } = req.body;
        const calendar = yield models_1.default.ExternalCalendar.findByPk(calendarId);
        if (!calendar) {
            return res.status(404).json({ message: 'External calendar not found' });
        }
        yield calendar.update({
            name: name !== undefined ? name : calendar.name,
            source: source !== undefined ? source : calendar.source,
            calendarUrl: calendarUrl !== undefined ? calendarUrl : calendar.calendarUrl,
            isActive: isActive !== undefined ? isActive : calendar.isActive
        });
        // If calendar URL changed, verify it works by syncing
        if (calendarUrl && calendarUrl !== calendar.calendarUrl) {
            try {
                yield (0, icalSync_1.syncExternalCalendars)(calendar.roomId);
                yield calendar.update({ lastSynced: new Date() });
            }
            catch (error) {
                console.error('Warning: Calendar updated but sync failed:', error);
            }
        }
        return res.status(200).json({
            status: 'success',
            data: { calendar }
        });
    }
    catch (error) {
        console.error('Error updating external calendar settings:', error);
        return res.status(500).json({ message: 'Failed to update calendar settings', error: error.message });
    }
});
exports.updateExternalCalendarSettings = updateExternalCalendarSettings;
