"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteBooking = exports.updateBooking = exports.createBooking = exports.getBookingById = exports.getAllBookings = void 0;
const sequelize_1 = require("sequelize");
const errorHandler_1 = require("../utils/errorHandler");
const models_1 = __importDefault(require("../models"));
const models_2 = require("../models");
const icalSync_1 = require("../utils/icalSync");
const feedback_controller_1 = require("./feedback.controller");
const email_service_1 = __importDefault(require("../services/email.service"));
// Utility to compute required minimum nights for a date range based on monthly rules
const getRequiredMinNightsForRange = (start, end) => __awaiter(void 0, void 0, void 0, function* () {
    // Build set of covered months from start..(end-1 day)
    const coveredMonths = new Set();
    try {
        const endExclusive = new Date(end);
        endExclusive.setDate(endExclusive.getDate() - 1);
        let cursor = new Date(Date.UTC(start.getUTCFullYear(), start.getUTCMonth(), 1));
        const endCursor = new Date(Date.UTC(endExclusive.getUTCFullYear(), endExclusive.getUTCMonth(), 1));
        while (cursor <= endCursor) {
            coveredMonths.add(cursor.getUTCMonth() + 1);
            cursor.setUTCMonth(cursor.getUTCMonth() + 1);
        }
        let requiredMin = 1;
        const rules = yield models_1.default.BookingRule.findAll();
        const monthToMin = new Map(rules.map((r) => [r.month, r.minNights]));
        for (const m of coveredMonths) {
            requiredMin = Math.max(requiredMin, monthToMin.get(m) || 1);
        }
        if (requiredMin > 20)
            requiredMin = 20;
        return requiredMin;
    }
    catch (err) {
        // If rules table not available or any error, default to 1
        return 1;
    }
});
// Helper function to check room availability for a date range
const isRoomAvailable = (roomId, checkInDate, checkOutDate, excludeBookingId) => __awaiter(void 0, void 0, void 0, function* () {
    const whereClause = {
        roomId,
        status: {
            [sequelize_1.Op.notIn]: ['cancelled'],
        },
        [sequelize_1.Op.or]: [
            {
                // Bookings that start during the requested period (excluding checkout date)
                checkInDate: {
                    [sequelize_1.Op.gte]: checkInDate,
                    [sequelize_1.Op.lt]: checkOutDate,
                },
            },
            {
                // Bookings that end during the requested period (excluding checkin date)
                checkOutDate: {
                    [sequelize_1.Op.gt]: checkInDate,
                    [sequelize_1.Op.lte]: checkOutDate,
                },
            },
            {
                // Bookings that span the entire requested period
                [sequelize_1.Op.and]: [
                    {
                        checkInDate: {
                            [sequelize_1.Op.lt]: checkInDate,
                        },
                    },
                    {
                        checkOutDate: {
                            [sequelize_1.Op.gt]: checkOutDate,
                        },
                    },
                ],
            },
        ],
    };
    // Exclude current booking for updates
    if (excludeBookingId) {
        whereClause.id = {
            [sequelize_1.Op.ne]: excludeBookingId,
        };
    }
    const conflictingBookings = yield models_1.default.Booking.count({
        where: whereClause,
    });
    return conflictingBookings === 0;
});
// Get all bookings with filters
exports.getAllBookings = (0, errorHandler_1.catchAsync)((req, res) => __awaiter(void 0, void 0, void 0, function* () {
    const { status, roomId, guestId, dateFrom, dateTo } = req.query;
    const whereClause = {};
    // Apply filters if provided
    if (status)
        whereClause.status = status;
    if (roomId)
        whereClause.roomId = roomId;
    if (guestId)
        whereClause.guestId = guestId;
    // Date range filter
    if (dateFrom || dateTo) {
        if (dateFrom) {
            whereClause.checkInDate = Object.assign(Object.assign({}, (whereClause.checkInDate || {})), { [sequelize_1.Op.gte]: new Date(dateFrom) });
        }
        if (dateTo) {
            whereClause.checkOutDate = Object.assign(Object.assign({}, (whereClause.checkOutDate || {})), { [sequelize_1.Op.lte]: new Date(dateTo) });
        }
    }
    const bookings = yield models_1.default.Booking.findAll({
        where: whereClause,
        include: [
            {
                model: models_1.default.Room,
                as: 'room',
                include: [
                    {
                        model: models_1.default.RoomType,
                        as: 'roomType',
                    },
                ],
            },
            {
                model: models_1.default.Guest,
                as: 'guest',
            },
            {
                model: models_1.default.User,
                as: 'creator',
                attributes: ['id', 'firstName', 'lastName', 'email'],
            },
            {
                model: models_1.default.Payment,
                as: 'payments',
            },
        ],
    });
    res.status(200).json({
        status: 'success',
        results: bookings.length,
        data: {
            bookings,
        },
    });
}));
// Get booking by ID
exports.getBookingById = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const booking = yield models_1.default.Booking.findByPk(req.params.id, {
        include: [
            {
                model: models_1.default.Room,
                as: 'room',
                include: [
                    {
                        model: models_1.default.RoomType,
                        as: 'roomType',
                    },
                ],
            },
            {
                model: models_1.default.Guest,
                as: 'guest',
            },
            {
                model: models_1.default.User,
                as: 'creator',
                attributes: ['id', 'firstName', 'lastName', 'email'],
            },
            {
                model: models_1.default.Payment,
                as: 'payments',
            },
        ],
    });
    if (!booking) {
        return next(new errorHandler_1.AppError('No booking found with that ID', 404));
    }
    res.status(200).json({
        status: 'success',
        data: {
            booking,
        },
    });
}));
// Create a new booking
exports.createBooking = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    // Extract required data from request body
    const { roomId, guestId, checkInDate, checkOutDate, specialRequests, } = req.body;
    // Parse dates
    const checkIn = new Date(checkInDate);
    const checkOut = new Date(checkOutDate);
    // Validate dates
    if (isNaN(checkIn.getTime()) || isNaN(checkOut.getTime())) {
        return next(new errorHandler_1.AppError('Invalid date format', 400));
    }
    if (checkIn >= checkOut) {
        return next(new errorHandler_1.AppError('Check-in date must be before check-out date', 400));
    }
    // Check if room exists and is not in maintenance
    const room = yield models_1.default.Room.findByPk(roomId);
    if (!room) {
        return next(new errorHandler_1.AppError('Room not found', 404));
    }
    // Check if guest exists
    const guest = yield models_1.default.Guest.findByPk(guestId);
    if (!guest) {
        return next(new errorHandler_1.AppError('Guest not found', 404));
    }
    // Check if room is available for the requested dates
    const isAvailable = yield isRoomAvailable(roomId, checkIn, checkOut);
    if (!isAvailable) {
        return next(new errorHandler_1.AppError('Room is not available for the selected dates', 400));
    }
    // Calculate number of nights
    const oneDay = 24 * 60 * 60 * 1000; // hours*minutes*seconds*milliseconds
    const nights = Math.round(Math.abs((checkOut.getTime() - checkIn.getTime()) / oneDay));
    // Enforce monthly minimum nights rule
    const requiredMin = yield getRequiredMinNightsForRange(checkIn, checkOut);
    if (nights < requiredMin) {
        return next(new errorHandler_1.AppError(`This stay requires a minimum of ${requiredMin} night(s) for the selected months due to property rules.`, 400));
    }
    // Calculate total price with discount
    const pricePerNight = room.pricePerNight;
    const discount = room.discount || 0;
    const totalPrice = nights * pricePerNight * (1 - (discount / 100));
    // Create booking using a transaction
    const result = yield models_2.sequelize.transaction((t) => __awaiter(void 0, void 0, void 0, function* () {
        // Create booking
        const booking = yield models_1.default.Booking.create({
            roomId,
            guestId,
            checkInDate: checkIn,
            checkOutDate: checkOut,
            status: 'pending',
            totalPrice,
            specialRequests,
            createdBy: req.user ? req.user.id : null,
        }, { transaction: t });
        // Update room availability
        const start = new Date(checkIn);
        const end = new Date(checkOut);
        end.setDate(end.getDate() - 1); // Exclude checkout day
        // Room availability is now booking-based only - no separate table needed
        // Update room status
        yield room.update({
            status: 'reserved',
        }, { transaction: t });
        return booking;
    }));
    // Include related models in the response
    const booking = yield models_1.default.Booking.findByPk(result.id, {
        include: [
            {
                model: models_1.default.Room,
                as: 'room',
                include: [
                    {
                        model: models_1.default.RoomType,
                        as: 'roomType',
                    },
                ],
            },
            {
                model: models_1.default.Guest,
                as: 'guest',
            },
        ],
    });
    // Sync external calendars for this room
    try {
        yield (0, icalSync_1.syncExternalCalendars)(booking.roomId);
    }
    catch (error) {
        // Don't fail the request if sync fails
    }
    res.status(201).json({
        status: 'success',
        data: {
            booking,
        },
    });
}));
// Update booking
exports.updateBooking = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    let booking;
    try {
        booking = yield models_1.default.Booking.findByPk(req.params.id);
        if (!booking) {
            return next(new errorHandler_1.AppError('No booking found with that ID', 404));
        }
    }
    catch (error) {
        throw error;
    }
    // Check if booking is already cancelled or completed
    if (['cancelled', 'completed'].includes(booking.status)) {
        return next(new errorHandler_1.AppError(`Cannot update a ${booking.status} booking`, 400));
    }
    const requestFields = {
        roomId: req.body.roomId,
        checkInDate: req.body.checkInDate,
        checkOutDate: req.body.checkOutDate,
        status: req.body.status,
        specialRequests: req.body.specialRequests
    };
    // Extract fields to update
    const { roomId, checkInDate, checkOutDate, status, specialRequests, } = req.body;
    let checkIn = booking.checkInDate;
    let checkOut = booking.checkOutDate;
    let room = yield models_1.default.Room.findByPk(booking.roomId);
    // If dates are being updated, validate them
    if (checkInDate || checkOutDate) {
        const finalCheckIn = new Date(checkInDate || booking.checkInDate);
        const finalCheckOut = new Date(checkOutDate || booking.checkOutDate);
        // Validate dates
        if (isNaN(finalCheckIn.getTime()) || isNaN(finalCheckOut.getTime())) {
            return next(new errorHandler_1.AppError('Invalid date format', 400));
        }
        if (finalCheckIn >= finalCheckOut) {
            return next(new errorHandler_1.AppError('Check-in date must be before check-out date', 400));
        }
        // If dates changed, need to check availability again
        // Convert dates to string format for comparison
        const currentCheckInStr = booking.checkInDate instanceof Date
            ? booking.checkInDate.toISOString().split('T')[0]
            : booking.checkInDate.toString();
        const currentCheckOutStr = booking.checkOutDate instanceof Date
            ? booking.checkOutDate.toISOString().split('T')[0]
            : booking.checkOutDate.toString();
        const newCheckInStr = checkInDate || currentCheckInStr;
        const newCheckOutStr = checkOutDate || currentCheckOutStr;
        // Apply the validated dates so subsequent logic (availability, pricing, update) uses them
        checkIn = finalCheckIn;
        checkOut = finalCheckOut;
        if (newCheckInStr !== currentCheckInStr || newCheckOutStr !== currentCheckOutStr) {
            const isAvailable = yield isRoomAvailable(roomId || booking.roomId, checkIn, checkOut, booking.id);
            if (!isAvailable) {
                return next(new errorHandler_1.AppError('Room is not available for the selected dates', 400));
            }
            // Enforce monthly minimum nights rule on updates
            const oneDayMs = 24 * 60 * 60 * 1000;
            const updatedNights = Math.round(Math.abs((checkOut.getTime() - checkIn.getTime()) / oneDayMs));
            const requiredMinUpdate = yield getRequiredMinNightsForRange(checkIn, checkOut);
            if (updatedNights < requiredMinUpdate) {
                return next(new errorHandler_1.AppError(`This stay requires a minimum of ${requiredMinUpdate} night(s) for the selected months due to property rules.`, 400));
            }
        }
    }
    // If room is being changed, check if the new room exists and is available
    if (roomId && roomId !== booking.roomId) {
        room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return next(new errorHandler_1.AppError('Room not found', 404));
        }
        if (room.status === 'maintenance') {
            return next(new errorHandler_1.AppError('Room is under maintenance and not available for booking', 400));
        }
        // Check if the new room is available
        const isAvailable = yield isRoomAvailable(roomId, checkIn, checkOut, booking.id);
        if (!isAvailable) {
            return next(new errorHandler_1.AppError('Room is not available for the selected dates', 400));
        }
    }
    // Update booking using a transaction
    yield models_2.sequelize.transaction((t) => __awaiter(void 0, void 0, void 0, function* () {
        // Calculate new totalPrice if room or dates changed
        let newTotalPrice = booking.totalPrice;
        if (roomId !== booking.roomId || checkInDate || checkOutDate) {
            // Use currently effective dates (already set above if provided)
            const finalCheckIn = new Date(checkIn);
            const finalCheckOut = new Date(checkOut);
            // Calculate number of nights
            const oneDay = 24 * 60 * 60 * 1000;
            const nights = Math.round(Math.abs((finalCheckOut.getTime() - finalCheckIn.getTime()) / oneDay));
            // Get room for price calculation
            const currentRoom = room || (yield models_1.default.Room.findByPk(booking.roomId, { transaction: t }));
            if (!currentRoom) {
                throw new errorHandler_1.AppError('Room not found for price calculation', 404);
            }
            const pricePerNight = currentRoom.pricePerNight;
            const discount = currentRoom.discount || 0;
            newTotalPrice = nights * pricePerNight * (1 - (discount / 100));
        }
        // First, if we're changing rooms or dates, update room availability
        if (roomId !== booking.roomId || checkInDate || checkOutDate) {
            // Room availability is now booking-based only - no separate table needed
            // If we're changing rooms, update the old room's status if needed
            if (roomId && roomId !== booking.roomId) {
                const oldRoom = yield models_1.default.Room.findByPk(booking.roomId, { transaction: t });
                if (oldRoom) {
                    // Check if the old room has other bookings
                    const otherBookings = yield models_1.default.Booking.count({
                        where: {
                            roomId: oldRoom.id,
                            id: {
                                [sequelize_1.Op.ne]: booking.id,
                            },
                            status: {
                                [sequelize_1.Op.in]: ['confirmed', 'pending'],
                            },
                        },
                        transaction: t,
                    });
                    // If no other bookings, set room status back to available
                    if (otherBookings === 0) {
                        yield oldRoom.update({
                            status: 'available',
                        }, { transaction: t });
                    }
                }
            }
        }
        // Update booking status
        if (status === 'cancelled') {
            // If cancelling, reset room availabilities and update room status
            // Room availability is now booking-based only - no separate table to clean up
            // Check if the room has other bookings
            const otherBookings = yield models_1.default.Booking.count({
                where: {
                    roomId: booking.roomId,
                    id: {
                        [sequelize_1.Op.ne]: booking.id,
                    },
                    status: {
                        [sequelize_1.Op.in]: ['confirmed', 'pending'],
                    },
                },
                transaction: t,
            });
            // If no other bookings, set room status back to available
            if (otherBookings === 0) {
                yield models_1.default.Room.update({
                    status: 'available',
                }, {
                    where: { id: booking.roomId },
                    transaction: t
                });
            }
        }
        else if (status === 'completed') {
            // When completing a booking, also reset availability
            // Room availability is now booking-based only - no separate table to clean up
            // Set room status to available when booking is completed
            yield models_1.default.Room.update({
                status: 'available',
            }, {
                where: { id: booking.roomId },
                transaction: t
            });
        }
        // Update booking
        const updateData = {
            roomId: roomId || booking.roomId,
            checkInDate: checkIn,
            checkOutDate: checkOut,
            status: status || booking.status,
            specialRequests: specialRequests !== undefined ? specialRequests : booking.specialRequests,
            totalPrice: newTotalPrice, // Update totalPrice
        };
        yield booking.update(updateData, { transaction: t });
    }));
    // Get the updated booking with related data
    const updatedBooking = yield models_1.default.Booking.findByPk(req.params.id, {
        include: [
            {
                model: models_1.default.Room,
                as: 'room',
                include: [
                    {
                        model: models_1.default.RoomType,
                        as: 'roomType',
                    },
                ],
            },
            {
                model: models_1.default.Guest,
                as: 'guest',
            },
        ],
    });
    // If booking was just completed, generate and send feedback request
    if (status === 'completed' && booking.status !== 'completed') {
        try {
            if (updatedBooking) {
                const feedbackToken = (0, feedback_controller_1.generateFeedbackToken)();
                const expiresAt = new Date();
                expiresAt.setDate(expiresAt.getDate() + 7); // Token expires in 7 days
                // Save the token to the database
                yield models_1.default.FeedbackToken.create({
                    token: feedbackToken,
                    bookingId: updatedBooking.id,
                    guestId: updatedBooking.guestId,
                    expiresAt,
                    isUsed: false,
                });
                const guest = updatedBooking.guest;
                const room = updatedBooking.room;
                const roomName = room.roomType ? room.roomType.name : `Room ${room.number}`;
                // Send the feedback request email with the token
                yield email_service_1.default.sendFeedbackRequest(guest.email, `${guest.firstName} ${guest.lastName}`, roomName, updatedBooking.checkInDate.toString(), updatedBooking.checkOutDate.toString(), feedbackToken);
            }
        }
        catch (error) {
            console.error('Failed to send feedback email or save token:', error);
            // We don't fail the entire request, but we log the error
        }
    }
    // Sync external calendars for this room
    try {
        yield (0, icalSync_1.syncExternalCalendars)(updatedBooking.roomId);
        // If room was changed, also sync the old room's calendar
        if (roomId && roomId !== booking.roomId) {
            yield (0, icalSync_1.syncExternalCalendars)(booking.roomId);
        }
    }
    catch (error) {
        // Don't fail the request if sync fails
    }
    res.status(200).json({
        status: 'success',
        data: {
            booking: updatedBooking,
        },
    });
}));
// Delete booking (admin only)
exports.deleteBooking = (0, errorHandler_1.catchAsync)((req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    const booking = yield models_1.default.Booking.findByPk(req.params.id);
    if (!booking) {
        return next(new errorHandler_1.AppError('No booking found with that ID', 404));
    }
    // Save room ID before deletion for calendar sync
    const roomId = booking.roomId;
    // Use transaction to ensure consistency
    yield models_2.sequelize.transaction((t) => __awaiter(void 0, void 0, void 0, function* () {
        // Room availability is now booking-based only - no separate table to clean up
        // Delete booking
        yield booking.destroy({ transaction: t });
        // Check if the room has other bookings
        const otherBookings = yield models_1.default.Booking.count({
            where: {
                roomId: booking.roomId,
                status: {
                    [sequelize_1.Op.in]: ['confirmed', 'pending'],
                },
            },
            transaction: t,
        });
        // If no other bookings, set room status back to available
        if (otherBookings === 0) {
            const room = yield models_1.default.Room.findByPk(booking.roomId, { transaction: t });
            if (room) {
                yield room.update({
                    status: 'available',
                }, { transaction: t });
            }
        }
    }));
    // Sync external calendars for this room
    try {
        yield (0, icalSync_1.syncExternalCalendars)(roomId);
    }
    catch (error) {
        // Don't fail the request if sync fails
    }
    res.status(204).json({
        status: 'success',
        data: null,
    });
}));
