"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.removeAmenityFromRoom = exports.addAmenityToRoom = exports.getRoomAmenities = exports.deleteAmenity = exports.updateAmenity = exports.createAmenity = exports.getAmenityById = exports.getAllAmenities = void 0;
const models_1 = __importDefault(require("../models"));
const errorHandler_1 = require("../utils/errorHandler");
const Amenity = models_1.default.Amenity;
// Get all amenities
const getAllAmenities = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const amenities = yield Amenity.findAll();
        return res.status(200).json({
            status: 'success',
            results: amenities.length,
            data: {
                amenities
            }
        });
    }
    catch (error) {
        next(error);
    }
});
exports.getAllAmenities = getAllAmenities;
// Get amenity by ID
const getAmenityById = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        const amenity = yield Amenity.findByPk(id);
        if (!amenity) {
            return next(new errorHandler_1.AppError(`Amenity with ID ${id} not found`, 404));
        }
        return res.status(200).json({
            status: 'success',
            data: {
                amenity
            }
        });
    }
    catch (error) {
        next(error);
    }
});
exports.getAmenityById = getAmenityById;
// Create a new amenity
const createAmenity = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const amenityData = req.body;
        const newAmenity = yield Amenity.create(amenityData);
        return res.status(201).json({
            status: 'success',
            data: {
                amenity: newAmenity
            }
        });
    }
    catch (error) {
        next(error);
    }
});
exports.createAmenity = createAmenity;
// Update an amenity
const updateAmenity = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        const updateData = req.body;
        const amenity = yield Amenity.findByPk(id);
        if (!amenity) {
            return next(new errorHandler_1.AppError(`Amenity with ID ${id} not found`, 404));
        }
        yield amenity.update(updateData);
        return res.status(200).json({
            status: 'success',
            data: {
                amenity
            }
        });
    }
    catch (error) {
        next(error);
    }
});
exports.updateAmenity = updateAmenity;
// Delete an amenity
const deleteAmenity = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { id } = req.params;
        const amenity = yield Amenity.findByPk(id);
        if (!amenity) {
            return next(new errorHandler_1.AppError(`Amenity with ID ${id} not found`, 404));
        }
        yield amenity.destroy();
        return res.status(204).json({
            status: 'success',
            data: null
        });
    }
    catch (error) {
        next(error);
    }
});
exports.deleteAmenity = deleteAmenity;
// Get all amenities for a room
const getRoomAmenities = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId } = req.params;
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return next(new errorHandler_1.AppError(`Room with ID ${roomId} not found`, 404));
        }
        const amenities = yield Amenity.findAll({
            include: [
                {
                    model: models_1.default.Room,
                    as: 'rooms',
                    where: { id: roomId },
                    through: { attributes: [] }
                }
            ]
        });
        return res.status(200).json({
            status: 'success',
            results: amenities.length,
            data: {
                amenities
            }
        });
    }
    catch (error) {
        next(error);
    }
});
exports.getRoomAmenities = getRoomAmenities;
// Add amenity to room
const addAmenityToRoom = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId, amenityId } = req.params;
        // Check if room exists
        const room = yield models_1.default.Room.findByPk(roomId);
        if (!room) {
            return next(new errorHandler_1.AppError(`Room with ID ${roomId} not found`, 404));
        }
        // Check if amenity exists
        const amenity = yield Amenity.findByPk(amenityId);
        if (!amenity) {
            return next(new errorHandler_1.AppError(`Amenity with ID ${amenityId} not found`, 404));
        }
        // Check if relationship already exists
        const existingRelation = yield models_1.default.RoomAmenity.findOne({
            where: { roomId, amenityId }
        });
        if (existingRelation) {
            return next(new errorHandler_1.AppError(`Amenity is already added to this room`, 400));
        }
        // Create relationship
        yield models_1.default.RoomAmenity.create({
            roomId,
            amenityId
        });
        return res.status(201).json({
            status: 'success',
            message: 'Amenity added to room successfully'
        });
    }
    catch (error) {
        next(error);
    }
});
exports.addAmenityToRoom = addAmenityToRoom;
// Remove amenity from room
const removeAmenityFromRoom = (req, res, next) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        const { roomId, amenityId } = req.params;
        // Check if relationship exists
        const relation = yield models_1.default.RoomAmenity.findOne({
            where: { roomId, amenityId }
        });
        if (!relation) {
            return next(new errorHandler_1.AppError(`Amenity is not associated with this room`, 404));
        }
        // Remove relationship
        yield relation.destroy();
        return res.status(200).json({
            status: 'success',
            message: 'Amenity removed from room successfully'
        });
    }
    catch (error) {
        next(error);
    }
});
exports.removeAmenityFromRoom = removeAmenityFromRoom;
